/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.contracts;

import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.SlaDto;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.model.contracts.ClientFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;

/**
 * Saves the last known {@link Snapshot} for each tracked {@link ApiContracts}.
 */
public class ContractSnapshots {

  private final ApiService apiService;
  private final Lock lock;
  private Map<ApiKey, Snapshot> snapshots = new HashMap<>();
  private final ClientFactory clientFactory;

  public ContractSnapshots(ApiService apiService, Lock lock, ClientFactory clientFactory) {
    this.apiService = apiService;
    this.lock = lock;
    this.clientFactory = clientFactory;
  }

  /**
   * Updates the current's {@link ApiContracts} {@link Snapshot} with the latest {@link Sla} information from Platform.
   * 
   * @param key {@link ApiKey}.
   * @param slas last adapted {@link SlaDto}s from Platform.
   * @return this.
   */
  public ContractSnapshots slas(ApiKey key, List<Sla> slas) {
    atomically(() -> update(key, snapshot(key).platformSlas(slas)));
    return this;
  }

  /**
   * Updates the current's {@link ApiContracts} {@link Snapshot} with the latest {@link Client} information from Platform.
   * 
   * @param key {@link ApiKey}.
   * @param platformContractAdapters last adapted {@link ApiClientDto} from Platform.
   * @return this.
   */
  public ContractSnapshots clients(ApiKey key, List<PlatformContractAdapter> platformContractAdapters) {
    atomically(() -> update(key, snapshot(key).platformContracts(platformContractAdapters)));
    return this;
  }

  private Snapshot snapshot(ApiKey key) {
    return snapshots
        .computeIfAbsent(key, apiKey -> apiService.get(key)
            .filter(api -> api.getContracts().contractsLoaded())
            .map(api -> new ContractSnapshot(api.getContracts().contracts(), clientFactory))
            .orElseGet(() -> new ContractSnapshot(clientFactory)));
  }

  private void update(ApiKey key, Snapshot newSnapshot) {
    snapshots.put(key, newSnapshot);
    apiService.getContracts(key).ifPresent(api -> {
      newSnapshot.contracts().ifPresent(api::updateContracts);
      newSnapshot.slas().ifPresent(api::updateSlas);
    });
  }

  private void atomically(Runnable closure) {
    lock.lock();
    closure.run();
    lock.unlock();
  }
}
