/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.service;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import com.mulesoft.mule.runtime.gw.api.client.ClientsRepository;
import com.mulesoft.mule.runtime.gw.api.client.ClientsRepositoryException;
import com.mulesoft.mule.runtime.gw.api.client.CoreServicesClient;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.dto.CoreServicesClientDto;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.response.PlatformResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link ClientsRepository} that queries Core Services for each requested {@link CoreServicesClient}.
 */
public class CoreServicesClientsRepository implements ClientsRepository {

  private static final Logger LOGGER = LoggerFactory.getLogger(CoreServicesClientsRepository.class);
  // TODO: AGW-5427: Replace patch placeholder with patch name
  private static final String PATCH_NAME = "<placeholder>";

  private ApiPlatformClient platform;

  public CoreServicesClientsRepository(ApiPlatformClient platform) {
    requireNonNull(platform, format("%s is null, cannot create the %s",
                                    ApiPlatformClient.class.getSimpleName(),
                                    CoreServicesClientsRepository.class.getSimpleName()));
    this.platform = platform;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CoreServicesClient getBy(String id) throws ClientsRepositoryException {
    throw new ClientsRepositoryException(format("Can not retrieve client using only client id. Please check you have patch '%s' correctly applied.",
                                                PATCH_NAME),
                                         400);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CoreServicesClient getBy(String id, String secret) throws ClientsRepositoryException {
    try {
      PlatformResponse<CoreServicesClientDto> platformResponse = platform.getClient(id, secret);
      if (platformResponse.statusCode() >= 400) {
        throw new ClientsRepositoryException(format("Invalid client with id '%s'.", id), platformResponse.statusCode());
      }

      CoreServicesClientDto dto = platformResponse.entity();
      return CoreServicesClient.getBuilder()
          .withId(dto.getId())
          .withSecret(dto.getSecret())
          .withName(dto.getName())
          .withRedirectionUris(dto.getRedirectUris())
          .build();
    } catch (HttpConnectionException e) {
      LOGGER.error("There was a connection error while looking for client with id '{}'. Cause: {}.", id, e.getMessage());
      throw new ClientsRepositoryException(format("Invalid client with id '%s'.", id), e.statusCode());
    } catch (HttpResponseException e) {
      LOGGER.error("Invalid status code returned, {}, when looking for client with id '{}'.", e.statusCode(), id);
      throw new ClientsRepositoryException(format("Invalid client with id '%s'.", id), e.statusCode());
    }
  }
}
