/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.contracts;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.contract.NoSla;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.SlaDto;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.model.contracts.ClientFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Defines the latest Snapshot State for an {@link ApiContracts}.
 */
public class ContractSnapshot implements Snapshot {

  private Optional<Map<Integer, Sla>> slas = empty();
  private Optional<List<Contract>> contracts = empty();
  private final ClientFactory clientFactory;

  public ContractSnapshot(ClientFactory clientFactory) {
    this.clientFactory = clientFactory;
  }

  public ContractSnapshot(List<Contract> storedContracts, ClientFactory clientFactory) {
    this.contracts = of(storedContracts);
    this.slas = of(storedContracts.stream()
        .map(Contract::sla)
        .filter(sla -> sla.id() != 0)
        .distinct()
        .collect(toMap(Sla::id, sla -> sla)));
    this.clientFactory = clientFactory;
  }

  /**
   * @return current available {@link Contract}s
   */
  @Override
  public Optional<List<Contract>> contracts() {
    return contracts
        .map(list -> list.stream()
            .map(contract -> newContract(contract.client(),
                                         slas.map(slas -> slas.getOrDefault(contract.sla().id(), new NoSla()))
                                             .orElse(new NoSla())))
            .collect(toList()));
  }

  @Override
  public Optional<List<Sla>> slas() {
    return slas.map(map -> new ArrayList<>(map.values()));
  }

  /**
   * Removes outdated {@link Sla}s, adds new ones and updates remaining ones.
   *
   * @param platformSlas last {@link SlaDto}s from Platform, adapted to our internal model.
   * @return this.
   */
  @Override
  public Snapshot platformSlas(List<Sla> platformSlas) {
    contracts = contracts.map(list -> list.stream()
        .filter(contract -> hasMatchingSla(contract, platformSlas))
        .collect(toList()));

    slas = of(platformSlas.stream().collect(toMap(Sla::id, sla -> sla)));

    return this;
  }

  /**
   * Removes outdated {@link Client}s, adds new ones and updates remaining ones.
   *
   * @param platformContractAdapters last {@link ApiClientDto}s from Platform, adapted to our internal model.
   * @return this.
   */
  @Override
  public Snapshot platformContracts(List<PlatformContractAdapter> platformContractAdapters) {
    this.contracts = of(platformContractAdapters.stream()
        .map(contract -> newContract(clientFactory.create(contract.clientId(), contract.clientSecret(), contract.clientName()),
                                     contract.slaId().isPresent() ? new Sla(contract.slaId().get()) : new NoSla()))
        .collect(toList()));
    return this;
  }

  /**
   * If a contract has an SLA defined, checks whether that SLA exists in Platform
   */
  private boolean hasMatchingSla(Contract contract, List<Sla> platformSlas) {
    return contract.sla().id() == 0
        || platformSlas.stream().anyMatch(platformSla -> platformSla.id().equals(contract.sla().id()));
  }

  private Contract newContract(Client client, Sla sla) {
    return Contract.builder().withClient(client).withSla(sla).build();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ContractSnapshot that = (ContractSnapshot) o;

    if (slas != null ? !slas.equals(that.slas) : that.slas != null) {
      return false;
    }
    return contracts != null ? contracts.equals(that.contracts) : that.contracts == null;
  }

  @Override
  public int hashCode() {
    int result = slas != null ? slas.hashCode() : 0;
    result = 31 * result + (contracts != null ? contracts.hashCode() : 0);
    return result;
  }
}
