/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment;

import static com.mulesoft.mule.runtime.gw.api.config.PlatformServicesConfiguration.KEEP_ALIVE_ENABLED;
import static com.mulesoft.mule.runtime.gw.api.config.PlatformServicesConfiguration.POLL_APIS_ENABLED;
import static com.mulesoft.mule.runtime.gw.api.config.PlatformServicesConfiguration.POLL_CLIENTS_ENABLED;
import static java.lang.System.clearProperty;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.mockito.ArgumentMatchers.nullable;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;

import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfigurationSupplier;
import com.mulesoft.anypoint.backoff.scheduler.BackoffScheduler;
import com.mulesoft.anypoint.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.model.ApiResponse;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientConnectionListener;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientProvider;
import com.mulesoft.mule.runtime.gw.client.response.PlatformResponse;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.PlatformInteractionServicesLifecycle;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.PlatformInteractionServicesLifecycleFactory;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;
import com.mulesoft.mule.runtime.gw.model.contracts.ApiContractsFactory;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;
import com.mulesoft.anypoint.retry.RunnableRetrier;
import com.mulesoft.mule.runtime.gw.retry.BackoffRunnableRetrierFactory;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;

public class PlatformInteractionServicesLifecycleTestCase {

  public static final String KEEP_ALIVE_POLL_NAME = "agw-api-keep-alive";
  public static final String CONTRACT_POLL_NAME = "agw-contract-polling";
  public static final String POLICY_POLL_NAME = "agw-policy-polling";

  private ApiService apiService;
  private Api api1;
  private Api api2;
  private Api api3;
  private BackoffSchedulerFactory schedulerFactory;
  private ApiPlatformClient restClient;
  private BackoffRunnableRetrierFactory backoffRunnableRetrierFactory;
  private BackoffConfigurationSupplier backoffConfigurationSupplier;
  private ApiTrackingService apiTrackingService;
  private MockLogger logger;
  private RunnableRetrier runnableRetrier;

  @Before
  public void setUp() {
    api1 = api(1L);
    api2 = api(2L);
    api3 = api(3L);
    apiService = mock(ApiService.class);
    when(apiService.getApis()).thenReturn(asList(api1, api2, api3));
    schedulerFactory = mock(BackoffSchedulerFactory.class);
    when(schedulerFactory.create(any())).thenReturn(mock(BackoffScheduler.class));
    restClient = mock(ApiPlatformClient.class);
    apiTrackingService = mock(ApiTrackingService.class);
    backoffConfigurationSupplier = mock(BackoffConfigurationSupplier.class);
    backoffRunnableRetrierFactory = mock(BackoffRunnableRetrierFactory.class);
    runnableRetrier = mock(RunnableRetrier.class);
    when(backoffRunnableRetrierFactory.apiTrackerRetrier()).thenReturn(runnableRetrier);
    logger = new MockLogger();
  }

  @After
  public void clean() {
    clearProperty(POLL_APIS_ENABLED);
    clearProperty(KEEP_ALIVE_ENABLED);
    clearProperty(POLL_CLIENTS_ENABLED);
  }

  @Test
  public void trackingSuccessfulSpoolSchedulers() {
    mockGetApiToReturn200();

    executeOnClientConnected();

    // That's as far as I can go as the ThreadPool and NamedThreadFactory does not have an equals.
    verify(schedulerFactory, times(3)).create(any(ScheduledExecutorService.class));
    verifyZeroInteractions(runnableRetrier);
  }

  @Test
  public void policyPollersNotScheduled() {
    mockGetApiToReturn200();

    System.setProperty(POLL_APIS_ENABLED, "false");
    executeOnClientConnected();

    assertPollersCreated(CONTRACT_POLL_NAME, KEEP_ALIVE_POLL_NAME);
  }

  @Test
  public void keepAliveNotScheduled() {
    mockGetApiToReturn200();

    System.setProperty(KEEP_ALIVE_ENABLED, "false");
    executeOnClientConnected();

    assertPollersCreated(CONTRACT_POLL_NAME, POLICY_POLL_NAME);
  }

  @Test
  public void clientsNotScheduled() {
    mockGetApiToReturn200();

    System.setProperty(POLL_CLIENTS_ENABLED, "false");
    executeOnClientConnected();

    assertPollersCreated(KEEP_ALIVE_POLL_NAME, POLICY_POLL_NAME);
  }

  @Test
  public void lastTrackingFailsButPollersAreSpooled() {
    mockGetApiToFailLast();

    executeOnClientConnected();

    // That's as far as I can go as the ThreadPool and NamedThreadFactory does not have an equals.
    verify(schedulerFactory, times(3)).create(any(ScheduledExecutorService.class));
    verify(runnableRetrier).scheduleRetry(eq(api3.getKey()), any());
    verifyNoMoreInteractions(runnableRetrier);
  }

  @Test
  public void someTrackerInTheMiddleFails() {
    mockGetApiToFailInTheMiddle();

    executeOnClientConnected();

    // That's as far as I can go as the ThreadPool and NamedThreadFactory does not have an equals.
    verify(schedulerFactory, times(3)).create(any(ScheduledExecutorService.class));
    verify(runnableRetrier).scheduleRetry(eq(api2.getKey()), any());
    verifyNoMoreInteractions(runnableRetrier);
  }

  private void executeOnClientConnected() {

    ApiPlatformClientProvider clientProvider = mock(ApiPlatformClientProvider.class);
    when(clientProvider.getClient()).thenReturn(restClient);

    PlatformInteractionServicesLifecycle platformManager =
        new PlatformInteractionServicesLifecycleFactory().create(apiService, apiTrackingService, clientProvider,
                                                                 schedulerFactory, backoffConfigurationSupplier,
                                                                 backoffRunnableRetrierFactory, true);

    ArgumentCaptor<ApiPlatformClientConnectionListener> captor =
        ArgumentCaptor.forClass(ApiPlatformClientConnectionListener.class);
    verify(clientProvider, atLeastOnce()).addConnectionListener(captor.capture());

    captor.getAllValues().forEach(ApiPlatformClientConnectionListener::onClientConnected);
  }

  private void mockGetApiToFailInTheMiddle() {
    ApiResponse apiResponse = mock(ApiResponse.class);
    when(apiResponse.getTrackingInfo()).thenReturn(mock(TrackingInfo.class));
    when(restClient.getApi(any(), nullable(String.class)))
        .thenReturn(new PlatformResponse<>(apiResponse, 200))
        .thenThrow(new OutOfMemoryError("throwing some crazy Error, may the gods have mercy on us all."))
        .thenReturn(new PlatformResponse<>(apiResponse, 401));
  }

  private void mockGetApiToFailLast() {
    ApiResponse apiResponse = mock(ApiResponse.class);
    when(apiResponse.getTrackingInfo()).thenReturn(mock(TrackingInfo.class));
    when(restClient.getApi(any(), nullable(String.class)))
        .thenReturn(new PlatformResponse<>(apiResponse, 200))
        .thenReturn(new PlatformResponse<>(apiResponse, 401))
        .thenThrow(new OutOfMemoryError("throwing some crazy Error, may the gods have mercy on us all."));
  }

  private void mockGetApiToReturn200() {
    ApiResponse apiResponse = mock(ApiResponse.class);
    when(apiResponse.getTrackingInfo()).thenReturn(mock(TrackingInfo.class));
    when(restClient.getApi(any(), nullable(String.class))).thenReturn(new PlatformResponse<>(apiResponse, 200));
  }

  private Api api(long apiId) {
    ApiKey key = new ApiKey(apiId);
    ApiImplementation implementation = mock(ApiImplementation.class);
    when(implementation.getApiKey()).thenReturn(key);
    return new Api(key, implementation, ApiContractsFactory.create(key, emptyList()));
  }

  private void assertPollersCreated(String... pollers) {
    ArgumentCaptor<ScheduledExecutorService> acScheduledExecutorService = ArgumentCaptor.forClass(ScheduledExecutorService.class);
    verify(schedulerFactory, times(pollers.length)).create(acScheduledExecutorService.capture());
    assertSchedulers(acScheduledExecutorService.getAllValues(), pollers);
  }

  private void assertSchedulers(List<ScheduledExecutorService> allValues, String... schedulers) {
    List<String> list = asList(schedulers);
    list.sort(Comparator.comparing(String::toString));

    List<String> names = allValues.stream()
        .map(scheduledExecutorService -> (String) new Inspector(scheduledExecutorService).read("threadFactory.name"))
        .sorted(Comparator.comparing(String::toString)).collect(Collectors.toList());

    assertThat(names, contains(list.toArray()));
  }

}
