/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.apis;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;

import com.mulesoft.anypoint.tests.logger.ErrorLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.anypoint.tests.logger.WarnLine;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.exception.IncompatibleApiException;
import com.mulesoft.mule.runtime.gw.client.exception.NotFoundException;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.model.ApiResponse;
import com.mulesoft.mule.runtime.gw.deployment.ApiPlatformTestCase;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ApisRunnable;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;

import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class ApisRunnableTestCase extends ApiPlatformTestCase {

  @Mock
  private PolicySet policySet;

  @Mock
  private Sla sla;

  @Mock
  private TrackingInfo trackingInfo;

  private ApisRunnable runnable;
  private MockLogger logger;

  @Override
  public void setUp() throws Exception {
    super.setUp();

    runnable = mocks.apisRunnable();
    logger = new MockLogger();
    overrideLogger().in(runnable).with(logger);
    overrideVariable("normalizedLogger").in(runnable).with(new RecoverableExceptionMessageLogger(logger));
  }

  @Test
  public void oneApiPolled() {
    List<Sla> slas = newArrayList(sla);
    ApiResponse apiResponse = new ApiResponse(trackingInfo, policySet, slas, true);
    mocks.singleTrackedApi()
        .getApiReturns(mocks.api(), apiResponse);

    runnable.run();

    verify(mocks.session()).getApi(mocks.api());
    verify(mocks.apiTrackingService()).getTrackedApis();
    verify(mocks.apiTrackingService()).apiTracked(API_KEY, trackingInfo, policySet, slas);
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void oneIncompatibleApi() {
    IncompatibleApiException exception = new IncompatibleApiException("");
    mocks.singleTrackedApi()
        .getApiThrows(mocks.api(), exception);

    runnable.run();

    verify(mocks.session()).getApi(mocks.api());
    verify(mocks.apiTrackingService()).getTrackedApis();
    verifyNoMoreInteractions(mocks.apiTrackingService());
    assertThat(logger.lines().contains(new WarnLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                                    "retrieve API information for API", mocks.api(),
                                                    errorMessage(exception))),
               is(true));
  }

  @Test
  public void connectionErrorGettingApi() {
    HttpConnectionException exception = new HttpConnectionException("message", null);
    mocks.singleTrackedApi()
        .getApiThrows(mocks.api(), exception);

    runnable.run();

    verify(mocks.session()).getApi(mocks.api());
    verify(mocks.apiTrackingService()).getTrackedApis();
    verifyNoMoreInteractions(mocks.apiTrackingService());
    assertThat(
               logger.lines()
                   .contains(new WarnLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                          "retrieve API information for API", mocks.api(),
                                          errorMessage(exception))),
               is(true));
  }

  @Test
  public void noApisPolled() {
    // when(mocks.apiTrackingService().getTrackedApis()).thenReturn(emptyList());

    runnable.run();

    verify(mocks.session(), never()).getApi(any(Api.class));
    verify(mocks.apiTrackingService()).getTrackedApis();
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void multipleApisPolled() {
    List<Sla> slas = newArrayList(sla);
    ApiResponse apiResponse = new ApiResponse(trackingInfo, policySet, slas, true);
    mocks.multipleTrackedApis()
        .getApiReturns(mocks.api(), apiResponse)
        .getApiReturns(mocks.secondApi(), apiResponse);

    runnable.run();

    verify(mocks.session()).getApi(mocks.api());
    verify(mocks.session()).getApi(mocks.secondApi());
    verify(mocks.apiTrackingService()).getTrackedApis();
    verify(mocks.apiTrackingService()).apiTracked(API_KEY, trackingInfo, policySet, slas);
    verify(mocks.apiTrackingService()).apiTracked(API_KEY_2, trackingInfo, policySet, slas);
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void policiesAreUntouchedIfNoUpdatesResponse() {
    List<Sla> slas = newArrayList(sla);
    ApiResponse apiResponse = new ApiResponse(trackingInfo, policySet, slas, false);
    mocks.singleTrackedApi()
        .getApiReturns(mocks.api(), apiResponse);

    runnable.run();

    verify(mocks.session()).getApi(mocks.api());
    verify(mocks.apiTrackingService()).getTrackedApis();
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void errorWhenRequestingPoliciesDoesNotBlocksOtherAPIs() {
    List<Sla> slas = newArrayList(sla);
    ApiResponse apiResponse = new ApiResponse(trackingInfo, policySet, slas, true);
    Exception exception = new HttpResponseException("");
    mocks.multipleTrackedApis()
        .getApiThrows(mocks.api(), exception)
        .getApiReturns(mocks.secondApi(), apiResponse);

    runnable.run();

    verify(mocks.session()).getApi(mocks.api());
    verify(mocks.session()).getApi(mocks.secondApi());
    verify(mocks.apiTrackingService()).getTrackedApis();
    verify(mocks.apiTrackingService()).apiTracked(eq(API_KEY_2), any(), any(), any());
    verifyNoMoreInteractions(mocks.apiTrackingService());
    assertThat(logger.lines().contains(
                                       new WarnLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                                    "retrieve API information for API", mocks.api(),
                                                    errorMessage(exception))),
               is(true));
  }

  @Test
  public void runDeletedApiFromPlatform() {
    mocks.singleTrackedApi()
        .getApiThrows(mocks.api(), new NotFoundException(""));

    runnable.run();

    verify(mocks.session()).getApi(mocks.api());
    verify(mocks.apiTrackingService()).getTrackedApis();
    verify(mocks.apiTrackingService()).apiUntracked(API_KEY);
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }
}
