/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.backoff;

import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.BACKOFF;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Arrays.asList;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.stream.IntStream.range;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import com.mulesoft.anypoint.tests.scheduler.ObservableScheduledExecutorService;
import com.mulesoft.anypoint.tests.scheduler.observer.ScheduledTask;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.anypoint.backoff.BackoffTestCase;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfigurationSupplier;
import com.mulesoft.anypoint.backoff.engine.BackoffSimulation;
import com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.anypoint.backoff.scheduler.factory.FixedExecutorBackoffSchedulerFactory;
import com.mulesoft.anypoint.backoff.scheduler.runnable.BackoffRunnable;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.deployment.backoff.mocks.SimulatedSessionStatusFactory;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.apis.GatewayApisPoller;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients.GatewayClientsPoller;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients.PlatformClientsRetriever;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.keepalive.GatewayKeepAlivePoller;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.Api;

import java.util.function.Function;

import org.junit.Before;
import org.junit.Test;

public class GatewayPollersBackoffTestCase extends BackoffTestCase {

  private Api api;
  private ApiTrackingService apiTrackingService;

  @Before
  public void setUp() {
    super.setUp();
    this.apiTrackingService = mock(ApiTrackingService.class);
    this.api = mock(Api.class, RETURNS_DEEP_STUBS);
    clearProperty(BACKOFF);
  }

  @Test
  public void noInteractionKeepAliveRemainsStable() {
    noInteractionRemainsStable(keepAliveConfiguration(), this::keepAliveRunnable);
  }

  @Test
  public void noInteractionApisRemainsStable() {
    noInteractionRemainsStable(apisConfiguration(), this::apisRunnable);
  }

  @Test
  public void noInteractionClientsRemainsStable() {
    noInteractionRemainsStable(clientsConfiguration(), this::clientsRunnable);
  }

  private void noInteractionRemainsStable(SchedulingConfiguration schedulingConfiguration,
                                          Function<SimulatedSessionStatusFactory, BackoffRunnable> backoffRunnableFactory) {
    int simulationIterations = 100;

    BackoffSimulation simulation = backoffSimulation(schedulingConfiguration).simulate(simulationIterations);

    SimulatedSessionStatusFactory platformFactory = simulatedPlatformSession(simulation);
    BackoffRunnable backoffRunnable = backoffRunnableFactory.apply(platformFactory);

    assertExecutionMatchesSimulation(backoffRunnable, simulation, platformFactory, simulationIterations);
  }

  @Test
  public void keepAliveBackoffBackon() {
    runBackoffBackonSimulation(keepAliveConfiguration(), this::noDispersionKeepAliveRunnable);
  }

  @Test
  public void apisBackoffBackon() {
    runBackoffBackonSimulation(apisConfiguration(), this::noDispersionApisRunnable);
  }

  @Test
  public void clientsBackoffBackon() {
    runBackoffBackonSimulation(clientsConfiguration(), this::noDispersionClientsRunnable);
  }

  private void runBackoffBackonSimulation(SchedulingConfiguration schedulingConfiguration,
                                          Function<SimulatedSessionStatusFactory, BackoffRunnable> backoffRunnableFactory) {
    BackoffSimulation simulation = backoffSimulation(schedulingConfiguration)
        .off(0, 7, 503)
        .simulate(11);

    assertExecutionMatchesSimulation(backoffRunnableFactory, simulation, 11);
  }

  @Test
  public void keepAliveDoNotBackoffOnAllErrors() {
    doNotBackoffOnAllErrors(keepAliveConfiguration(), this::noDispersionKeepAliveRunnable);
  }

  @Test
  public void apisDoNotBackoffOnAllErrors() {
    doNotBackoffOnAllErrors(apisConfiguration(), this::noDispersionApisRunnable);
  }

  @Test
  public void clientsDoNotBackoffOnAllErrors() {
    doNotBackoffOnAllErrors(clientsConfiguration(), this::noDispersionClientsRunnable);
  }

  private void doNotBackoffOnAllErrors(SchedulingConfiguration schedulingConfiguration,
                                       Function<SimulatedSessionStatusFactory, BackoffRunnable> backoffRunnableFactory) {
    BackoffSimulation simulation = backoffSimulation(schedulingConfiguration)
        .off(0, 77, 418)
        .disabled() // As 418 is not a backoff status, we expected no backoff delay.
        .simulate(100);

    assertExecutionMatchesSimulation(backoffRunnableFactory, simulation, 100);
  }

  @Test
  public void keepAliveHalfUpFullDown() {
    halfUpFullDown(keepAliveConfiguration(), this::noDispersionKeepAliveRunnable);
  }

  @Test
  public void apisHalfUpFullDown() {
    halfUpFullDown(apisConfiguration(), this::noDispersionApisRunnable);
  }

  @Test
  public void clientsHalfUpFullDown() {
    halfUpFullDown(clientsConfiguration(), this::noDispersionClientsRunnable);
  }

  private void halfUpFullDown(SchedulingConfiguration schedulingConfiguration,
                              Function<SimulatedSessionStatusFactory, BackoffRunnable> backoffRunnableFactory) {
    BackoffSimulation simulation = backoffSimulation(schedulingConfiguration)
        .off(0, 3, 503)
        .simulate(20);

    assertExecutionMatchesSimulation(backoffRunnableFactory, simulation, 20);
  }


  @Test
  public void keepAliveOscillation() {
    oscillation(keepAliveConfiguration(), this::noDispersionKeepAliveRunnable);
  }

  @Test
  public void apisOscillation() {
    oscillation(apisConfiguration(), this::noDispersionApisRunnable);
  }

  @Test
  public void clientsOscillation() {
    oscillation(clientsConfiguration(), this::noDispersionClientsRunnable);
  }

  private void oscillation(SchedulingConfiguration schedulingConfiguration,
                           Function<SimulatedSessionStatusFactory, BackoffRunnable> backoffRunnableFactory) {
    BackoffSimulation simulation = backoffSimulation(schedulingConfiguration)
        .off(0, 2, 503)
        .off(4, 6, 503)
        .off(8, 12, 503)
        .off(14, 15, 503)
        .off(16, 37, 503)
        .off(38, 41, 503)
        .simulate(60);

    assertExecutionMatchesSimulation(backoffRunnableFactory, simulation, 60);
  }

  @Test
  public void backoffDisabled() {
    setProperty(BACKOFF, "false");

    BackoffSimulation simulation = backoffSimulation(keepAliveConfiguration())
        .off(0, 50, 503)
        .disabled()
        .simulate(50);

    assertExecutionMatchesSimulation(this::keepAliveRunnable, simulation, 50);
  }

  private BackoffRunnable noDispersionApisRunnable(SimulatedSessionStatusFactory platformFactory) {
    return withNoDispersion(apisPollersManager(platformFactory).schedule());
  }

  private BackoffRunnable noDispersionClientsRunnable(SimulatedSessionStatusFactory platformFactory) {
    return withNoDispersion(clientsPollersManager(platformFactory).schedule());
  }

  private BackoffRunnable noDispersionKeepAliveRunnable(SimulatedSessionStatusFactory platformFactory) {
    return withNoDispersion(keepAlivePollersManager(platformFactory).schedule());
  }

  private void trackAnApi() {
    when(apiTrackingService.getTrackedApis()).thenReturn(asList(api));
    when(apiTrackingService.getTrackedApisRequiringContracts()).thenReturn(asList(api));
    when(api.getImplementation().getFlow().getMuleContext().isStarted()).thenReturn(true);
  }

  private BackoffRunnable keepAliveRunnable(SimulatedSessionStatusFactory platformFactory) {
    return keepAlivePollersManager(platformFactory).schedule();
  }

  private BackoffRunnable apisRunnable(SimulatedSessionStatusFactory platformFactory) {
    return apisPollersManager(platformFactory).schedule();
  }

  private BackoffRunnable clientsRunnable(SimulatedSessionStatusFactory platformFactory) {
    return clientsPollersManager(platformFactory).schedule();
  }

  private void assertExecutionMatchesSimulation(Function<SimulatedSessionStatusFactory, BackoffRunnable> backoffRunnableFactory,
                                                BackoffSimulation simulation, int simulationIterations) {
    trackAnApi();

    SimulatedSessionStatusFactory platformFactory = simulatedPlatformSession(simulation);
    BackoffRunnable backoffRunnable = backoffRunnableFactory.apply(platformFactory);

    assertExecutionMatchesSimulation(backoffRunnable, simulation, platformFactory, simulationIterations);
  }

  private void assertExecutionMatchesSimulation(BackoffRunnable backoffRunnable, BackoffSimulation simulation,
                                                SimulatedSessionStatusFactory platformFactory, int simulationIterations) {
    range(0, simulationIterations).forEach(i -> {
      platformFactory.iteration(i);

      scheduledRunnable(i).run();

      assertThat(iteration(i), executorLogger.scheduledTasks().get(i),
                 is(new ScheduledTask(backoffRunnable, simulation.delay(i), 0, MILLISECONDS)));
    });
  }

  private GatewayApisPoller apisPollersManager(ApiPlatformSessionFactory platformSessionFactory) {
    return new GatewayApisPoller(new GatewayConfiguration(),
                                 apiTrackingService,
                                 platformSessionFactory,
                                 backoffSchedulerFactory(),
                                 new BackoffConfigurationSupplier());
  }

  private GatewayClientsPoller clientsPollersManager(ApiPlatformSessionFactory platformSessionFactory) {
    return new GatewayClientsPoller(new GatewayConfiguration(),
                                    apiTrackingService,
                                    platformSessionFactory,
                                    backoffSchedulerFactory(),
                                    new BackoffConfigurationSupplier(),
                                    new PlatformClientsRetriever(platformSessionFactory, apiTrackingService));
  }

  private GatewayKeepAlivePoller keepAlivePollersManager(ApiPlatformSessionFactory platformSessionFactory) {
    return new GatewayKeepAlivePoller(new GatewayConfiguration(),
                                      apiTrackingService,
                                      platformSessionFactory,
                                      backoffSchedulerFactory(),
                                      new BackoffConfigurationSupplier());
  }

  private BackoffSimulation backoffSimulation(SchedulingConfiguration schedulingConfiguration) {
    return new BackoffSimulation(schedulingConfiguration.delay(),
                                 schedulingConfiguration.frequency(),
                                 withNoDispersion(backoffConfiguration()));
  }

  private SimulatedSessionStatusFactory simulatedPlatformSession(BackoffSimulation simulation) {
    return new SimulatedSessionStatusFactory(simulation);
  }

  private BackoffSchedulerFactory backoffSchedulerFactory() {
    return new FixedExecutorBackoffSchedulerFactory(new ObservableScheduledExecutorService(executorLogger));
  }

  private SchedulingConfiguration keepAliveConfiguration() {
    return new GatewayKeepAlivePoller(new GatewayConfiguration(), null, null, null, null).configuration();
  }

  private SchedulingConfiguration apisConfiguration() {
    return new GatewayApisPoller(new GatewayConfiguration(), null, null, null, null).configuration();
  }

  private SchedulingConfiguration clientsConfiguration() {
    return new GatewayClientsPoller(new GatewayConfiguration(), null, null, null, null, null).configuration();
  }

  /**
   * Remove backoff/on dispersion in a {@link BackoffRunnable} as having it only adds testing complexity. It has already been
   * tested in its own unit tests.
   */
  private BackoffRunnable withNoDispersion(BackoffRunnable backoffRunnable) {
    BackoffConfiguration configuration = read(backoffRunnable, "currentState.configuration");
    overrideVariable("currentState.configuration").in(backoffRunnable).with(withNoDispersion(configuration));
    return backoffRunnable;
  }

  private BackoffConfiguration backoffConfiguration() {
    return new BackoffConfiguration.Builder(true).build();
  }

}
