/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.contracts;

import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;

import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.anypoint.tests.logger.WarnLine;
import com.mulesoft.mule.runtime.gw.client.adapter.ApiClientsResponseBuilder;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.model.ApiClientsResponse;
import com.mulesoft.mule.runtime.gw.deployment.ApiPlatformTestCase;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ClientsRunnable;
import com.mulesoft.mule.runtime.gw.model.Api;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

public class ContractSnapshotsRunnablesTestCase extends ApiPlatformTestCase {

  private ClientsRunnable runnable;
  private MockLogger logger;

  @Before
  public void setUp() throws Exception {
    super.setUp();

    runnable = mocks.clientsRunnable();
    logger = new MockLogger();
    overrideLogger().in(runnable).with(logger);
    overrideVariable("retriever.normalizedLogger").in(runnable).with(new RecoverableExceptionMessageLogger(logger));
  }

  @Test
  public void noClients() throws IOException, URISyntaxException {
    ApiClientsResponse apiClientsResponse = apiClientsResponse(mocks.api(), emptyList());
    mocks
        .singleApiFetchContracts()
        .getApiClientsReturns(mocks.api(), apiClientsResponse);

    runnable.run();

    verify(mocks.apiTrackingService()).getTrackedApisRequiringContracts();
    verify(mocks.apiTrackingService()).clientsPolling(mocks.api().getKey(), emptyList(),
                                                      apiClientsResponse.getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.api().getTrackingInfo().getOrganizationId(),
                                          mocks.api().getTrackingInfo().getEnvironmentId(),
                                          mocks.api().getTrackingInfo().getId(),
                                          mocks.api().getTrackingInfo().getContractsEntityTag());
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void oneClient() throws IOException {
    ApiClientDto client = apiClient(0, SLA_ID);
    ApiClientsResponse apiClientsResponse = apiClientsResponse(mocks.api(), singletonList(client));
    mocks
        .singleApiFetchContracts()
        .getApiClientsReturns(mocks.api(), apiClientsResponse);

    runnable.run();

    verify(mocks.apiTrackingService()).getTrackedApisRequiringContracts();
    verify(mocks.apiTrackingService()).clientsPolling(mocks.api().getKey(), singletonList(new PlatformContractAdapter(client)),
                                                      apiClientsResponse.getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.api().getTrackingInfo().getOrganizationId(),
                                          mocks.api().getTrackingInfo().getEnvironmentId(),
                                          mocks.api().getTrackingInfo().getId(),
                                          mocks.api().getTrackingInfo().getContractsEntityTag());
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void multipleClients() throws IOException {
    ApiClientDto client = apiClient(0, SLA_ID);
    ApiClientDto client2 = apiClient(1, ANOTHER_SLA_ID);
    ApiClientsResponse apiClientsResponse = apiClientsResponse(mocks.api(), asList(client, client2));
    mocks
        .singleApiFetchContracts()
        .getApiClientsReturns(mocks.api(), apiClientsResponse);

    runnable.run();

    verify(mocks.apiTrackingService()).getTrackedApisRequiringContracts();
    verify(mocks.apiTrackingService()).clientsPolling(mocks.api().getKey(),
                                                      asList(new PlatformContractAdapter(client),
                                                             new PlatformContractAdapter(client2)),
                                                      apiClientsResponse.getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.api().getTrackingInfo().getOrganizationId(),
                                          mocks.api().getTrackingInfo().getEnvironmentId(),
                                          mocks.api().getTrackingInfo().getId(),
                                          mocks.api().getTrackingInfo().getContractsEntityTag());
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  /**
   * If no policy requires contract service to fetch contracts no request shall be made to getApiClients.
   */
  @Test
  public void notFetchingContracts() throws IOException {
    mocks.singleTrackedApi();

    runnable.run();

    verify(mocks.apiTrackingService()).getTrackedApisRequiringContracts();
    verify(mocks.session(), never()).getApiClients(any(), any(), any());
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void multipleApisFetchingContracts() throws IOException {
    ApiClientDto client = apiClient(0, SLA_ID);
    ApiClientsResponse apiClientsResponse = apiClientsResponse(mocks.api(), singletonList(client));
    ApiClientsResponse secondApiClientsResponse = apiClientsResponse(mocks.secondApi(), singletonList(client));
    mocks.multipleApisFetchContracts()
        .getApiClientsReturns(mocks.api(), apiClientsResponse)
        .getApiClientsReturns(mocks.secondApi(), secondApiClientsResponse);

    runnable.run();

    verify(mocks.apiTrackingService()).getTrackedApisRequiringContracts();
    verify(mocks.apiTrackingService()).clientsPolling(mocks.api().getKey(), singletonList(new PlatformContractAdapter(client)),
                                                      apiClientsResponse.getContractsEntityTag());
    verify(mocks.apiTrackingService()).clientsPolling(mocks.secondApi().getKey(),
                                                      singletonList(new PlatformContractAdapter(client)),
                                                      secondApiClientsResponse.getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.api().getTrackingInfo().getOrganizationId(),
                                          mocks.api().getTrackingInfo().getEnvironmentId(),
                                          mocks.api().getTrackingInfo().getId(),
                                          mocks.api().getTrackingInfo().getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.secondApi().getTrackingInfo().getOrganizationId(),
                                          mocks.secondApi().getTrackingInfo().getEnvironmentId(),
                                          mocks.secondApi().getTrackingInfo().getId(),
                                          mocks.secondApi().getTrackingInfo().getContractsEntityTag());
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  @Test
  public void processAllAPIsDespiteGetApiClientFailing() throws IOException {
    ApiClientDto client = apiClient(0, SLA_ID);
    Exception exception = new HttpResponseException("");
    ApiClientsResponse secondApiClientsResponse = apiClientsResponse(mocks.secondApi(), singletonList(client));
    mocks.multipleApisFetchContracts()
        .getApiClientsThrows(mocks.api(), exception)
        .getApiClientsReturns(mocks.secondApi(), secondApiClientsResponse);

    runnable.run();

    verify(mocks.apiTrackingService()).getTrackedApisRequiringContracts();
    verify(mocks.apiTrackingService()).clientsPolling(mocks.secondApi().getKey(),
                                                      singletonList(new PlatformContractAdapter(client)),
                                                      secondApiClientsResponse.getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.api().getTrackingInfo().getOrganizationId(),
                                          mocks.api().getTrackingInfo().getEnvironmentId(),
                                          mocks.api().getTrackingInfo().getId(),
                                          mocks.api().getTrackingInfo().getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.secondApi().getTrackingInfo().getOrganizationId(),
                                          mocks.secondApi().getTrackingInfo().getEnvironmentId(),
                                          mocks.secondApi().getTrackingInfo().getId(),
                                          mocks.secondApi().getTrackingInfo().getContractsEntityTag());
    verifyNoMoreInteractions(mocks.apiTrackingService());
    assertThat(logger.lines().contains(
                                       new WarnLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                                    "obtain contracts for API",
                                                    mocks.api(),
                                                    errorMessage(exception))),
               is(true));
  }

  @Test
  public void processAllAPIsDespiteGetApiClientConnectionFailing() throws IOException {
    ApiClientDto client = apiClient(0, SLA_ID);
    Exception exception = new HttpConnectionException("message", null);
    ApiClientsResponse secondApiClientsResponse = apiClientsResponse(mocks.secondApi(), singletonList(client));
    mocks.multipleApisFetchContracts()
        .getApiClientsThrows(mocks.api(), exception)
        .getApiClientsReturns(mocks.secondApi(), secondApiClientsResponse);

    runnable.run();

    verify(mocks.apiTrackingService()).getTrackedApisRequiringContracts();
    verify(mocks.apiTrackingService()).clientsPolling(mocks.secondApi().getKey(),
                                                      singletonList(new PlatformContractAdapter(client)),
                                                      secondApiClientsResponse.getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.api().getTrackingInfo().getOrganizationId(),
                                          mocks.api().getTrackingInfo().getEnvironmentId(),
                                          mocks.api().getTrackingInfo().getId(),
                                          mocks.api().getTrackingInfo().getContractsEntityTag());
    verify(mocks.session()).getApiClients(mocks.secondApi().getTrackingInfo().getOrganizationId(),
                                          mocks.secondApi().getTrackingInfo().getEnvironmentId(),
                                          mocks.secondApi().getTrackingInfo().getId(),
                                          mocks.secondApi().getTrackingInfo().getContractsEntityTag());
    verifyNoMoreInteractions(mocks.apiTrackingService());
    assertThat(logger.lines().contains(
                                       new WarnLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                                    "obtain contracts for API",
                                                    mocks.api(),
                                                    errorMessage(exception))),
               is(true));
  }

  private ApiClientDto apiClient(int id, int slaId) {
    return dtoClient(id, mocks.api().getTrackingInfo().getId(), String.valueOf(slaId));
  }

  private ApiClientsResponse apiClientsResponse(Api api, List<ApiClientDto> clients) {
    return new ApiClientsResponseBuilder().withClients(clients).withContractsEntityTag("aContractsEntityTag" + api.getKey().id())
        .build();
  }

}
