/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.contracts;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.mule.runtime.gw.api.contract.Contract.builder;
import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.verifyZeroInteractions;

import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.deployment.mocks.BlowUpLock;
import com.mulesoft.mule.runtime.gw.model.contracts.DefaultClientFactory;

import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class ContractSnapshotsUnitTestCase extends ContractSnapshotsTestCase {

  @Test
  public void oneTierOneClient() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void oneClientOneTier() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void updateTier() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(anotherSla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contract(anotherSla(SLA_ID), client(SLA_ID))));
  }

  @Test
  public void updateClient() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(anotherPlatformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contract(sla(SLA_ID), anotherClient(SLA_ID))));
  }

  @Test
  public void consecutiveIdempotentTierUpdate() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void interleavedIdempotentTierUpdate() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void firstIdempotentTierUpdate() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              EMPTY_UPDATE,
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void consecutiveIdempotentClientUpdate() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void interleavedIdempotentClientUpdate() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)));


    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void firstIdempotentClientUpdate() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void unmatchedTierId() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              EMPTY_UPDATE);
  }

  @Test
  public void unmatchedClientTierId() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(noSlaContractForClient(ANOTHER_SLA_ID)));
  }

  @Test
  public void slaWithMultipleLimits() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(slaWithMultipleLimits(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractMultiLimit(SLA_ID)));
  }

  @Test
  public void multipleTiersOneClient() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(slaWithMultipleLimits(ANOTHER_SLA_ID), sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void multipleClientsOneTier() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID), platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(ANOTHER_SLA_ID),
                                                     noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void multipleTiersAndClients() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID), platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(ANOTHER_SLA_ID),
                                                     noSlaContractForClient(SLA_ID)),
                                              asList(contractMultiLimit(ANOTHER_SLA_ID),
                                                     contractSingleLimit(SLA_ID)));
  }

  @Test
  public void clearContractsWhenClientsRemoved() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID), platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList());

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(ANOTHER_SLA_ID),
                                                     noSlaContractForClient(SLA_ID)),
                                              asList(contractMultiLimit(ANOTHER_SLA_ID),
                                                     contractSingleLimit(SLA_ID)),
                                              asList());
  }

  @Test
  public void removeClientWithSla() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID), platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(ANOTHER_SLA_ID),
                                                     noSlaContractForClient(SLA_ID)),
                                              asList(contractMultiLimit(ANOTHER_SLA_ID),
                                                     contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void removeClientWithNoSla() {
    String noSla = null;
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(noSla)))
        .clients(apiKey(), asList(platformContractAdapter(noSla)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(noSla)),
                                              asList(noSlaContractForClient(noSla)));
  }

  @Test
  public void removeClientWithAndWithoutSla() {
    String noSla = null;
    snapshots
        .clients(apiKey(),
                 asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(noSla),
                        aThirdPlatformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(noSla)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(noSla),
                                                     noSlaContractForThirdClient(SLA_ID)),
                                              asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(noSla),
                                                     contract(sla(SLA_ID), aThirdClient(SLA_ID))),
                                              asList(noSlaContractForClient(noSla)));
  }

  @Test
  public void removeClientWithNoSlaButLeaveAnotherWithSla() {
    String noSla = null;
    snapshots
        .clients(apiKey(),
                 asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(noSla),
                        aThirdPlatformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(noSla), aThirdPlatformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(noSla),
                                                     noSlaContractForThirdClient(SLA_ID)),
                                              asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(noSla),
                                                     contract(sla(SLA_ID), aThirdClient(SLA_ID))),
                                              asList(noSlaContractForClient(noSla),
                                                     contract(sla(SLA_ID), aThirdClient(SLA_ID))));
  }

  @Test
  public void revokeClientWithNoSla() {
    String noSla = null;
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(anotherPlatformContractAdapter(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(SLA_ID)),
                                              asList(noSlaContractForClient(noSla),
                                                     contract(sla(SLA_ID), anotherClient(SLA_ID))),
                                              asList(contract(sla(SLA_ID), anotherClient(SLA_ID))),
                                              asList(noSlaContractForClient(noSla),
                                                     contract(sla(SLA_ID), anotherClient(SLA_ID))));
  }

  @Test
  public void revokeClientWithSla() {
    String noSla = null;
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .slas(apiKey(), asList(sla(ANOTHER_SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(SLA_ID)),
                                              asList(noSlaContractForClient(noSla),
                                                     contract(sla(SLA_ID), anotherClient(SLA_ID))),
                                              asList(noSlaContractForClient(noSla)),
                                              asList(noSlaContractForClient(noSla)),
                                              asList(noSlaContractForClient(noSla),
                                                     contract(sla(SLA_ID), anotherClient(SLA_ID))));

  }

  @Test
  public void addClientsWithNoSla() {
    String noSla = null;
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(noSla)))
        .clients(apiKey(), asList(platformContractAdapter(noSla), anotherPlatformContractAdapter(noSla)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(noSla)),
                                              asList(noSlaContractForClient(noSla),
                                                     noSlaContractForAnotherClient(noSla)));
  }

  @Test
  public void clientObtainsSla() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(null)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(null)),
                                              asList(noSlaContractForClient(null)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void onlyOneClientChangesSla() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID), anotherPlatformContractAdapter(ANOTHER_SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID), anotherPlatformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID),
                                                     noSlaContractForAnotherClient(ANOTHER_SLA_ID)),
                                              asList(contract(sla(SLA_ID), client(SLA_ID)),
                                                     contract(slaWithMultipleLimits(ANOTHER_SLA_ID),
                                                              anotherClient(ANOTHER_SLA_ID))),
                                              asList(contract(sla(SLA_ID), client(SLA_ID)),
                                                     contract(sla(SLA_ID), anotherClient(SLA_ID))));
  }

  @Test
  public void multipleClientsSameTier() {
    snapshots
        .clients(apiKey(), asList(anotherPlatformContractAdapter(SLA_ID), platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForAnotherClient(SLA_ID),
                                                     noSlaContractForClient(SLA_ID)),
                                              asList(contract(sla(SLA_ID), anotherClient(SLA_ID)),
                                                     contractSingleLimit(SLA_ID)));
  }

  @Test
  public void callTiersOrClientWithInvalidApiKey() {
    snapshots
        .clients(inexistentApi(), asList(platformContractAdapter(SLA_ID)))
        .slas(inexistentApi(), asList(sla(SLA_ID)));

    verify(mocks.apiService(), times(2)).getContracts(inexistentApi());
    verifyApiTestInvocations();
  }

  @Test
  public void clearContractsWhenTiersRemoved() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID), platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList());

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractMultiLimit(ANOTHER_SLA_ID),
                                                     contractSingleLimit(SLA_ID)),
                                              asList());
  }

  @Test
  public void deltaContractsWhenSomeTiersRemoved() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID), platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractMultiLimit(ANOTHER_SLA_ID),
                                                     contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)));
  }

  @Test
  public void clientChangesTier() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID), slaWithMultipleLimits(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractMultiLimit(ANOTHER_SLA_ID)));
  }

  @Test
  public void clientFromNoContractToChangingItTwice() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(null)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), sla(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(null)),
                                              asList(noSlaContractForClient(null)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(ANOTHER_SLA_ID)));
  }

  @Test
  public void clientFromNoContractToChangingItTwiceDifferentOrder() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(null)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), sla(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(null)),
                                              asList(noSlaContractForClient(null)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(ANOTHER_SLA_ID)));
  }

  @Test
  public void evolveClientFromNothingToSlaBackToNothing() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(null)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), sla(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID)))
        .slas(apiKey(), asList())
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(anotherSla(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(ANOTHER_SLA_ID)),
                                              EMPTY_UPDATE,
                                              asList(noSlaContractForClient(SLA_ID)),
                                              asList(contract(anotherSla(SLA_ID), client(SLA_ID))));
  }

  @Test
  public void tierChangesClient() {
    snapshots
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .clients(apiKey(), asList(anotherPlatformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(EMPTY_UPDATE,
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contract(sla(SLA_ID), anotherClient(SLA_ID))));
  }

  @Test
  public void slasUsesLock() {
    snapshots.slas(apiKey(), asList());
    verify(lock).lock();
    verify(lock).unlock();
  }

  @Test
  public void clientsUsesLock() {
    snapshots.clients(apiKey(), asList());
    verify(lock).lock();
    verify(lock).unlock();
  }

  @Test
  public void ifLockThrowsExceptionTiersClosureDoesNotRun() {
    BlowUpLock blowUpLock = new BlowUpLock();
    Runnable callTiers = () -> new ContractSnapshots(mocks.apiService(), blowUpLock, new DefaultClientFactory())
        .slas(apiKey(), asList(sla(SLA_ID)));
    whenLockBlowsUpClosureIsNotExecuted(blowUpLock, callTiers);
  }

  @Test
  public void whenLockThrowsExceptionClientsClosureIsNotExecuted() {
    BlowUpLock blowUpLock = new BlowUpLock();
    Runnable callClients = () -> new ContractSnapshots(mocks.apiService(), blowUpLock, new DefaultClientFactory())
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)));
    whenLockBlowsUpClosureIsNotExecuted(blowUpLock, callClients);
  }

  @Test
  public void sameClientChangingTiers() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID), sla(ANOTHER_SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(contractSingleLimit(ANOTHER_SLA_ID)));
  }

  @Test
  public void clientChangesTierAndTierAddedAfterwards() {
    snapshots
        .clients(apiKey(), asList(platformContractAdapter(SLA_ID)))
        .slas(apiKey(), asList(sla(SLA_ID)))
        .clients(apiKey(), asList(platformContractAdapter(ANOTHER_SLA_ID)))
        .slas(apiKey(), asList(sla(ANOTHER_SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(noSlaContractForClient(SLA_ID)),
                                              asList(contractSingleLimit(SLA_ID)),
                                              asList(noSlaContractForClient(ANOTHER_SLA_ID)),
                                              asList(contractSingleLimit(ANOTHER_SLA_ID)));
  }

  @Test
  public void clientPollsUpdatesWithContractsWithNoTiers() {
    snapshots.clients(apiKey(), asList(platformContractAdapter(null), anotherPlatformContractAdapter(SLA_ID)));

    checkContractsUpdatesAndMockedInvocations(asList(builder().withClient(client(null)).build(),
                                                     noSlaContractForAnotherClient(SLA_ID)));
  }

  private void whenLockBlowsUpClosureIsNotExecuted(BlowUpLock blowUpLock, Runnable closure) {
    boolean blowUpLockExploded = false;
    try {
      closure.run();
    } catch (RuntimeException e) {
      blowUpLockExploded = true;
    }

    if (!blowUpLockExploded) {
      fail("Blow up lock didn't explode");
    }
    verifyZeroInteractions(mocks.apiTrackingService());
    verifyApiTestInvocations();
    assertThat(blowUpLock.isUnlockedCalled(), is(false));
  }

  private void checkContractsUpdatesAndMockedInvocations(List<Contract>... expectedUpdates) {
    verify(mocks.apiService(), times(expectedUpdates.length)).getContracts(eq(apiKey()));
    checkContractsUpdates(expectedUpdates);
    verifyApiTestInvocations();
    verifyNoMoreInteractions(mocks.apiTrackingService());
  }

  private ApiKey apiKey() {
    return API_KEY;
  }
}
