/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.keep.alive;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.deployment.mocks.ApiPlatformMocks.ENV_ID;
import static com.mulesoft.mule.runtime.gw.deployment.mocks.ApiPlatformMocks.ORG_ID;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;

import com.mulesoft.anypoint.tests.logger.ErrorLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.anypoint.tests.logger.WarnLine;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.exception.NotFoundException;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.deployment.ApiPlatformTestCase;
import com.mulesoft.mule.runtime.gw.deployment.runnable.ApiKeepAliveRunnable;

import java.io.IOException;
import java.net.URISyntaxException;

import org.junit.Test;

public class ApiKeepAliveRunnableTestCase extends ApiPlatformTestCase {

  private ApiKeepAliveRunnable instance;
  private MockLogger logger;

  @Override
  public void setUp() throws Exception {
    super.setUp();

    instance = mocks.keepAliveRunnable();
    logger = new MockLogger();
    overrideLogger().in(instance).with(logger);
    overrideVariable("normalizedLogger").in(instance).with(new RecoverableExceptionMessageLogger(logger));
  }

  @Test
  public void runCompletesSuccessfully() throws IOException {
    mocks.singleTrackedApi();

    instance.run();

    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID);
    verifyRemainingSessionInteractions();
  }

  @Test
  public void runCompletesSuccessfullyMultipleApis() throws IOException {
    mocks.multipleTrackedApis();

    instance.run();

    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID);
    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID_2);
    verifyRemainingSessionInteractions();
  }

  @Test
  public void runNoEndpointToActivate() throws IOException {
    mocks.singleTrackedApi().stoppedApi();

    instance.run();

    verifyRemainingSessionInteractions();
  }

  @Test
  public void runThrowsException() throws IOException {
    doThrow(new RuntimeException()).when(mocks.session()).activateEndpoint(any(), any(), any());

    instance.run();

    verifyRemainingSessionInteractions();
  }

  @Test
  public void exceptionForAnApiDoesNotBlocksOtherApis() throws IOException, URISyntaxException {
    mocks.multipleTrackedApis();

    doThrow(new RuntimeException()).when(mocks.session()).activateEndpoint(any(), any(), eq(API_ID));

    instance.run();

    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID);
    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID_2);
    verifyRemainingSessionInteractions();
  }

  @Test
  public void runDeletedApiFromPlatform() throws IOException, URISyntaxException {
    mocks.singleTrackedApi();

    doThrow(new NotFoundException("")).when(mocks.session()).activateEndpoint(any(), any(), any());

    instance.run();

    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID);
    verify(mocks.apiTrackingService()).apiUntracked(API_KEY);
    verifyRemainingSessionInteractions();
  }

  @Test
  public void unsuccessfulResponseWhenConnectingToPlatform() {
    mocks.singleTrackedApi();
    Exception e = new HttpResponseException("");
    doThrow(e).when(mocks.session()).activateEndpoint(any(), any(), eq(API_ID));

    instance.run();

    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID);
    verifyRemainingSessionInteractions();
    assertThat(logger.lines().contains(
                                       new WarnLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                                    "perform keep alive request for API",
                                                    mocks.api(), errorMessage(e))),
               is(true));
  }

  @Test
  public void connectionErrorConnectingToPlatform() {
    mocks.singleTrackedApi();
    Exception e = new HttpConnectionException("message", null);
    doThrow(e).when(mocks.session()).activateEndpoint(any(), any(), eq(API_ID));

    instance.run();

    verify(mocks.session()).activateEndpoint(ORG_ID, ENV_ID, API_ID);
    verifyRemainingSessionInteractions();
    assertThat(logger.lines().contains(
                                       new WarnLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                                    "perform keep alive request for API",
                                                    mocks.api(), errorMessage(e))),
               is(true));
  }

  private void verifyRemainingSessionInteractions() {
    verify(mocks.session()).metadata();
    verifyNoMoreInteractions(mocks.session());
  }
}
