/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.replication;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.mule.runtime.gw.model.PolicySet.PolicySetOrigin.PLATFORM;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;

import org.mule.runtime.core.internal.serialization.JavaExternalSerializerProtocol;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.contracts.ContractSnapshots;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;
import com.mulesoft.mule.runtime.module.cluster.internal.serialization.ClusterDistributedObject;

import com.hazelcast.core.EntryEvent;

import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

public class DistributedPoliciesMapEntryListenerTestCase {

  private static final int OVERRIDEN_HASCODE = 1902;
  private DistributedPoliciesMapEntryListener entryListener;
  private MockLogger logger;
  private ApiService apiService;
  private ApiTrackingService apiTrackingService;
  private ContractSnapshots contractSnapshots;
  private PolicyDefinition policyDefinition;
  private PolicySetDeploymentService policySetDeploymentService;

  @Before
  public void setUp() {
    apiService = mock(ApiService.class);
    policySetDeploymentService = mock(PolicySetDeploymentService.class);
    contractSnapshots = mock(ContractSnapshots.class);
    logger = new MockLogger();
    policyDefinition = mock(PolicyDefinition.class);
    apiTrackingService = mock(ApiTrackingService.class);
    entryListener = new DistributedPoliciesMapEntryListener(apiService, apiTrackingService,
                                                            policySetDeploymentService, contractSnapshots, false);

    overrideLogger().in(entryListener).with(logger);
  }

  @Test
  public void entryAddedLocalMember() {
    EntryEvent entry = mockEntry(true);

    entryListener.entryAdded(entry);

    verifyNoInteractions();
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0), is(entryAddedDebugLine()));
  }

  @Test
  public void deployApiEntryAddedNonLocalMember() {
    when(apiService.isDeployed(any())).thenReturn(true);
    EntryEvent<String, DistributedApiConfigurationEntry> entry = mockEntry(false);

    entryListener.entryAdded(entry);

    assertWithDeployedApi(entry);
    assertStoringLogs();
  }

  @Test
  public void notDeployedApiEntryAddedNonLocalMember() {
    when(apiService.isDeployed(any())).thenReturn(false);
    EntryEvent<String, DistributedApiConfigurationEntry> entry = mockEntry(false);

    entryListener.entryAdded(entry);

    assertWithNotDeployedApi(entry);
    assertStoringLogs();
  }

  @Test
  public void entryUpdateLocalMember() {
    EntryEvent entry = mockEntry(true);

    entryListener.entryUpdated(entry);

    verifyNoInteractions();
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0), is(entryModifiedDebugLine()));
  }

  @Test
  public void deployApiEntryUpdatedNonLocalMember() {
    when(apiService.isDeployed(any())).thenReturn(true);
    EntryEvent<String, DistributedApiConfigurationEntry> entry = mockEntry(false);

    entryListener.entryUpdated(entry);

    assertWithDeployedApi(entry);
    assertUpdatingLogs();
  }

  @Test
  public void notDeployedApiEntryUpdatedNonLocalMember() {
    when(apiService.isDeployed(any())).thenReturn(false);
    EntryEvent<String, DistributedApiConfigurationEntry> entry = mockEntry(false);

    entryListener.entryUpdated(entry);

    assertWithNotDeployedApi(entry);
    assertUpdatingLogs();
  }

  @Test
  public void onEntryRemovedLocalMember() {
    EntryEvent entry = mockEntry(true);

    entryListener.entryRemoved(entry);

    verify(entry, times(2)).getMember();
    verify(entry.getMember()).localMember();
    verify(entry, times(2)).getKey();
    verifyZeroInteractions(apiTrackingService);
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0), is(entryRemovedDebugLine()));
    assertThat(logger.lines().get(1), is(firedRemovalDebugLine()));
  }

  @Test
  public void onEntryRemovedNonLocalMember() {
    EntryEvent entry = mockEntry(false);

    entryListener.entryRemoved(entry);

    verify(entry, times(2)).getMember();
    verify(entry.getMember()).localMember();
    verify(entry, times(3)).getKey();
    verify(apiTrackingService).apiUntracked(API_KEY);
    verifyNoMoreInteractions(apiTrackingService);
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0), is(entryRemovedDebugLine()));
    assertThat(logger.lines().get(1), is(removingEntryDebugLine()));
  }

  private EntryEvent<String, DistributedApiConfigurationEntry> mockEntry(boolean isLocalMember) {
    EntryEvent entry = mock(EntryEvent.class, RETURNS_DEEP_STUBS);
    when(entry.getKey()).thenReturn(API_KEY.id().toString());
    when(entry.getValue()).thenReturn(apiConfigurationEntry());
    when(entry.getMember().localMember()).thenReturn(isLocalMember);
    return entry;
  }

  private void verifyNoInteractions() {
    verifyZeroInteractions(contractSnapshots);
    verifyZeroInteractions(apiTrackingService);
    verifyZeroInteractions(apiService);
    verifyZeroInteractions(policySetDeploymentService);
  }

  private void assertStoringLogs() {
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0), is(entryAddedDebugLine()));
    assertThat(logger.lines().get(1), is(entryStoringDebugLine()));
  }

  private void assertUpdatingLogs() {
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0), is(entryModifiedDebugLine()));
    assertThat(logger.lines().get(1), is(entryBackupDebugLine()));
  }

  private void assertWithDeployedApi(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    ApiKey apiKey = API_KEY;
    PolicySet policySet = entry.getValue().getPolicySet();
    List<Sla> slas = entry.getValue().getSlas();
    verify(apiService).isDeployed(apiKey);
    verify(contractSnapshots).slas(eq(apiKey), eq(slas));
    verify(policySetDeploymentService).policiesForApi(eq(apiKey), eq(policySet));
    verifyNoMoreInteractions(apiService);
    verifyNoMoreInteractions(contractSnapshots);
    verifyNoMoreInteractions(policySetDeploymentService);
    verifyZeroInteractions(apiTrackingService);
  }

  private void assertWithNotDeployedApi(EntryEvent<String, DistributedApiConfigurationEntry> entry) {
    ApiKey apiKey = API_KEY;
    List<PolicyDefinition> policyDefinitions = entry.getValue().getPolicySet().getPolicyDefinitions();
    verify(apiService).isDeployed(apiKey);
    verify(policySetDeploymentService).conciliatePolicies(eq(apiKey), eq(policyDefinitions));
    verifyNoMoreInteractions(apiService);
    verifyNoMoreInteractions(policySetDeploymentService);
    verifyZeroInteractions(apiTrackingService);
    verifyZeroInteractions(contractSnapshots);
  }

  private DistributedApiConfigurationEntry apiConfigurationEntry() {
    return new DistributedApiConfigurationEntry(new PolicySet(asList(policyDefinition), PLATFORM), new ArrayList<>()) {

      private static final long serialVersionUID = -7490827916319887757L;

      @Override
      public int hashCode() {
        return OVERRIDEN_HASCODE;
      }
    };
  }

  private DebugLine removingEntryDebugLine() {
    return new DebugLine("Removing map entry [{}] as the requester is not us.", API_KEY);
  }

  private DebugLine entryRemovedDebugLine() {
    return new DebugLine("Policies map entry removed [{}].", API_KEY);
  }

  private DebugLine firedRemovalDebugLine() {
    return new DebugLine("Local member fired the removal of [{}].", API_KEY);
  }

  private DebugLine entryAddedDebugLine() {
    return new DebugLine("Policies map entry added [{}] by {} [hashCode={}]", API_KEY, null, OVERRIDEN_HASCODE);
  }

  private DebugLine entryStoringDebugLine() {
    return new DebugLine("Storing map entry [{}] into map store as a backup", API_KEY);
  }

  private DebugLine entryModifiedDebugLine() {
    return new DebugLine("Policies map entry updated [{}] by {} [hashCode={}]", API_KEY, null, OVERRIDEN_HASCODE);
  }

  private DebugLine entryBackupDebugLine() {
    return new DebugLine("Updating map entry [{}] into map store as a backup", API_KEY);
  }
}
