/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.tracking;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.mule.runtime.gw.model.NoTrackingInfo.trackingFailed;
import static com.mulesoft.mule.runtime.gw.model.NoTrackingInfo.untracked;
import static java.util.Optional.of;
import static org.hamcrest.Matchers.contains;
import static org.junit.Assert.assertThat;
import static org.mockito.Answers.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.OnApiDeletedConfiguration;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.contracts.ContractSnapshots;
import com.mulesoft.mule.runtime.gw.deployment.replication.ApiConfigurationCache;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;

import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class DefaultApiTrackingServiceTestCase extends AbstractMuleTestCase {

  @Mock
  private ApiService apiService;

  @Mock
  private PolicySetDeploymentService policySetDeploymentService;

  @Mock
  private ApiConfigurationCache apiConfigurationCache;

  @Mock
  private ContractSnapshots contractSnapshots;

  @Mock
  private ContractService contractService;

  @Mock(answer = RETURNS_DEEP_STUBS)
  private Api api;

  @Mock(answer = RETURNS_DEEP_STUBS)
  private Api anotherApi;

  @Mock
  private Api untrackedApi;

  @Mock
  private Api failedApi;

  private ApiTrackingService service;

  @Before
  public void setUp() {
    service = new DefaultApiTrackingService(apiService, policySetDeploymentService, apiConfigurationCache,
                                            contractSnapshots, contractService,
                                            new OnApiDeletedConfiguration(new GateKeeperConfiguration()));

    when(apiService.get(API_KEY)).thenReturn(of(api));
    when(api.getKey()).thenReturn(API_KEY);
    when(api.getTrackingInfo().isTracked()).thenReturn(true);
    when(anotherApi.getTrackingInfo().isTracked()).thenReturn(true);
    when(untrackedApi.getTrackingInfo()).thenReturn(untracked());
    when(failedApi.getTrackingInfo()).thenReturn(trackingFailed());
  }

  @Test
  public void apiTracked() {
    TrackingInfo trackingInfo = mock(TrackingInfo.class);
    PolicySet policySet = mock(PolicySet.class);
    List<Sla> slas = newArrayList(mock(Sla.class));

    service.apiTracked(API_KEY, trackingInfo, policySet, slas);

    verify(api).updateTrackingInfo(trackingInfo);
    verify(apiConfigurationCache).set(API_KEY, policySet, slas);
    verify(policySetDeploymentService).policiesForApi(API_KEY, policySet);
    verify(contractSnapshots).slas(API_KEY, slas);
  }

  @Test
  public void apiUntrackedAndBlocked() {
    try {
      System.setProperty("anypoint.platform.policies.on_api_deleted", "BLOCK_API");
      service.apiUntracked(API_KEY);

      verify(api).updateTrackingInfo(untracked());
      verify(apiConfigurationCache).remove(API_KEY);
      verify(policySetDeploymentService).removeAll(API_KEY);
    } finally {
      System.clearProperty("anypoint.platform.policies.on_api_deleted");
    }
  }

  @Test
  public void apiUntrackedAndKept() {
    PolicySet policySet = mock(PolicySet.class);
    when(apiConfigurationCache.getPolicies(API_KEY)).thenReturn(of(policySet));

    service.apiUntracked(API_KEY);

    verify(api).updateTrackingInfo(untracked());
    verify(policySetDeploymentService).policiesForApi(API_KEY, policySet);
  }

  @Test
  public void apiTrackingFailed() {
    PolicySet policySet = mock(PolicySet.class);
    when(apiConfigurationCache.getPolicies(API_KEY)).thenReturn(of(policySet));
    when(api.getTrackingInfo()).thenReturn(untracked());

    service.apiTrackingFailed(API_KEY);

    verify(api).updateTrackingInfo(trackingFailed());
    verify(policySetDeploymentService).policiesForApi(API_KEY, policySet);
  }

  @Test
  public void apiTrackingFailedAfterItAlreadyFailed() {
    PolicySet policySet = mock(PolicySet.class);
    when(apiConfigurationCache.getPolicies(API_KEY)).thenReturn(of(policySet));
    when(api.getTrackingInfo()).thenReturn(trackingFailed());

    service.apiTrackingFailed(API_KEY);

    verify(api, never()).updateTrackingInfo(trackingFailed());
    verify(policySetDeploymentService, never()).policiesForApi(API_KEY, policySet);
  }

  @Test
  public void getTrackedApis() {
    when(apiService.getApis()).thenReturn(newArrayList(api, anotherApi, untrackedApi, failedApi));

    List<Api> trackedApis = service.getTrackedApis();

    assertThat(trackedApis, contains(api, anotherApi));
  }

  @Test
  public void getFailedApis() {
    when(apiService.getApis()).thenReturn(newArrayList(api, anotherApi, untrackedApi, failedApi));

    List<Api> failedApis = service.getFailedTrackingApis();

    assertThat(failedApis, contains(failedApi));
  }

  @Test
  public void getTrackedApisRequiringContracts() {
    ApiKey anotherApiKey = mock(ApiKey.class);
    ApiKey untrackedApiKey = mock(ApiKey.class);
    ApiKey failedApiKey = mock(ApiKey.class);

    when(anotherApi.getKey()).thenReturn(anotherApiKey);
    when(untrackedApi.getKey()).thenReturn(untrackedApiKey);
    when(failedApi.getKey()).thenReturn(failedApiKey);

    when(contractService.trackedApis()).thenReturn(newArrayList(API_KEY, untrackedApiKey, failedApiKey));
    when(apiService.getApis()).thenReturn(newArrayList(api, anotherApi, untrackedApi, failedApi));

    List<Api> requiringContracts = service.getTrackedApisRequiringContracts();

    assertThat(requiringContracts, contains(api));
  }

}
