/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.tracking;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.anypoint.tests.logger.ErrorLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.exception.IncompatibleApiException;
import com.mulesoft.mule.runtime.gw.client.exception.NotFoundException;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.model.ApiResponse;
import com.mulesoft.mule.runtime.gw.client.session.ApiPlatformSession;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;
import com.mulesoft.anypoint.retry.RunnableRetrier;

import java.io.IOException;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class RetriableApiTrackerTestCase extends AbstractMuleTestCase {

  @Mock
  private ApiTrackingService apiTrackingService;

  @Mock
  private RunnableRetrier<ApiKey> runnableRetrier;

  @Mock
  private ApiPlatformSessionFactory sessionFactory;

  @Mock
  private ApiPlatformSession session;

  @Mock
  private ApiImplementation implementation;

  @Mock
  private Api api;

  private RetriableApiTracker apiTracker;
  private MockLogger logger;

  @Before
  public void setUp() {
    ApiTracker tracker = new ApiTracker(apiTrackingService, sessionFactory);
    apiTracker = new RetriableApiTracker(tracker, runnableRetrier);

    when(sessionFactory.create()).thenReturn(session);
    when(implementation.getApiKey()).thenReturn(API_KEY);
    when(api.getImplementation()).thenReturn(implementation);

    logger = new MockLogger();

    overrideLogger().in(tracker).with(logger);
    overrideVariable("normalizedLogger").in(tracker).with(new RecoverableExceptionMessageLogger(logger));
  }

  @Test
  public void success() {
    TrackingInfo trackingInfo = mock(TrackingInfo.class);
    PolicySet policySet = mock(PolicySet.class);
    List<Sla> slas = newArrayList(mock(Sla.class));
    ApiResponse apiResponse = new ApiResponse(trackingInfo, policySet, slas, true);
    when(session.getApi(any(ApiKey.class))).thenReturn(apiResponse);

    apiTracker.onApiDeploymentSuccess(api);

    verify(apiTrackingService).apiTracked(API_KEY, trackingInfo, policySet, slas);
    verifyNoMoreInteractions(apiTrackingService);
    verifyZeroInteractions(runnableRetrier);
  }

  @Test
  public void incompatibleApi() throws IOException {
    IncompatibleApiException exception = new IncompatibleApiException("");
    when(session.getApi(any(ApiKey.class))).thenThrow(exception);

    apiTracker.onApiDeploymentSuccess(api);

    verifyNoMoreInteractions(apiTrackingService);
    verifyZeroInteractions(runnableRetrier, apiTrackingService);
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(1),
               is(new ErrorLine("There was an error trying to track API {}. {}", 1L, "Reason: " + exception)));
  }

  @Test
  public void apiNotFound() {
    when(session.getApi(API_KEY)).thenThrow(new NotFoundException(""));

    apiTracker.onApiDeploymentSuccess(api);

    verify(apiTrackingService).apiUntracked(API_KEY);
    verifyNoMoreInteractions(apiTrackingService);
    verifyZeroInteractions(runnableRetrier);
  }

  @Test
  public void retryIsScheduledAfterFailure() {
    HttpResponseException exception = new HttpResponseException("");
    when(session.getApi(API_KEY)).thenThrow(exception);

    apiTracker.onApiDeploymentSuccess(api);

    verify(apiTrackingService).apiTrackingFailed(API_KEY);
    verify(runnableRetrier).scheduleRetry(any(), any());
    verifyNoMoreInteractions(apiTrackingService);
    assertThat(logger.lines()
        .contains(new ErrorLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                "retrieve API information for API", API_KEY.id(), errorMessage(exception))),
               is(true));
  }

  @Test
  public void retryIsScheduledAfterConnectionFailure() {
    HttpConnectionException exception = new HttpConnectionException("message", null);
    when(session.getApi(API_KEY)).thenThrow(exception);

    apiTracker.onApiDeploymentSuccess(api);

    verify(apiTrackingService).apiTrackingFailed(API_KEY);
    verify(runnableRetrier).scheduleRetry(any(), any());
    verifyNoMoreInteractions(apiTrackingService);
    assertThat(logger.lines()
        .contains(new ErrorLine("Failed to {} '{}'. This request will be retried after some backoff time. {}",
                                "retrieve API information for API", API_KEY.id(), errorMessage(exception))),
               is(true));
  }

  @Test
  public void retryIsScheduledAfterError() {
    when(session.getApi(API_KEY)).thenThrow(new InternalError());

    apiTracker.onApiDeploymentSuccess(api);

    verify(apiTrackingService).apiTrackingFailed(API_KEY);
    verify(runnableRetrier).scheduleRetry(any(), any());
    verifyNoMoreInteractions(apiTrackingService);
  }

}
