/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.runnable;

import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;

import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.session.BackoffBarrier;
import com.mulesoft.anypoint.backoff.session.SessionMetadata;
import com.mulesoft.mule.runtime.gw.client.exception.NormalizedExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.exception.NotFoundException;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.model.ApiResponse;
import com.mulesoft.mule.runtime.gw.client.session.ApiPlatformSession;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;

import java.util.Optional;

import org.slf4j.Logger;

public class ApisRunnable extends GatewayPollerRunnable {

  private static final String GET_API_DESCRIPTION = "retrieve API information for API";
  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(ApisRunnable.class);

  private final NormalizedExceptionMessageLogger normalizedLogger = new RecoverableExceptionMessageLogger(LOGGER);

  public ApisRunnable(ApiTrackingService apiTrackingService,
                      ApiPlatformSessionFactory platformSessionFactory,
                      BackoffBarrier backoffBarrier,
                      BackoffConfiguration backoffConfiguration) {
    super(apiTrackingService, platformSessionFactory, backoffBarrier, backoffConfiguration);
  }

  @Override
  protected SessionMetadata execute() {
    return profiledExecution(() -> {
      LOGGER.debug("Collecting sources across all applications");

      ApiPlatformSession platform = platformSessionFactory.create();
      apiTrackingService.getTrackedApis().forEach(api -> {
        LOGGER.debug("Fetching policies for {}", api);
        try {
          getApi(api, platform).ifPresent(apiResponse -> {
            if (apiResponse.hasUpdates()) {
              LOGGER.debug("Retrieved {} policies for API {}", apiResponse.countPolicies(), api);

              apiTrackingService.apiTracked(api.getKey(), apiResponse.getTrackingInfo(), apiResponse.getPolicySet(),
                                            apiResponse.getSlas());
            }
          });
        } catch (Exception e) {
          LOGGER.error("Unexpected error occurred. {}", errorMessage(e));
        }
      });

      return platform.metadata();
    });
  }

  private Optional<ApiResponse> getApi(Api api, ApiPlatformSession platform) {
    try {
      return ofNullable(platform.getApi(api));
    } catch (NotFoundException nfe) {
      apiTrackingService.apiUntracked(api.getKey());
    } catch (Exception e) {
      normalizedLogger.warn(GET_API_DESCRIPTION, api, e);
    } catch (Throwable e) {
      LOGGER.error(format("Unexpected error occurred retrieving API %s. %s", api, errorMessage(e)), e);
      throw e;
    }

    return empty();
  }

}
