/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.notification;

import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;

import org.slf4j.Logger;

public class ApiNotificationManager {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(ApiNotificationManager.class);

  private List<ApiDeploymentListener> apiDeploymentListeners = new CopyOnWriteArrayList<>();

  public void notifyApiDeploymentStart(Api api) {
    notify(deploymentStart(api), "API Deployment start");
  }

  public void notifyApiDeploymentSuccess(Api api) {
    notify(deploymentSuccess(api), "API Deployment success");
  }

  public void notifyApiUndeploymentStart(ApiImplementation implementation) {
    notify(undeploymentStart(implementation), "API Undeployment start");
  }

  public void notifyApiRedeploymentStart(ApiImplementation implementation) {
    notify(redeploymentStart(implementation), "API Redeployment start");
  }

  public void addApiDeploymentListener(ApiDeploymentListener apiDeploymentListener) {
    apiDeploymentListeners.add(apiDeploymentListener);
  }

  public void removeApiDeploymentListener(ApiDeploymentListener apiDeploymentListener) {
    apiDeploymentListeners.remove(apiDeploymentListener);
  }

  private void notify(Consumer<ApiDeploymentListener> notification, String description) {
    apiDeploymentListeners
        .forEach(listener -> {
          try {
            notification.accept(listener);
          } catch (Exception e) {
            LOGGER.warn("Error on {} listener: {}", description, e.getMessage());
          }
        });
  }

  private Consumer<ApiDeploymentListener> deploymentStart(Api api) {
    return listener -> listener.onApiDeploymentStart(api);
  }

  private Consumer<ApiDeploymentListener> deploymentSuccess(Api api) {
    return listener -> listener.onApiDeploymentSuccess(api);
  }

  private Consumer<ApiDeploymentListener> undeploymentStart(ApiImplementation implementation) {
    return listener -> listener.onApiUndeploymentStart(implementation);
  }

  private Consumer<ApiDeploymentListener> redeploymentStart(ApiImplementation implementation) {
    return listener -> listener.onApiRedeploymentStart(implementation);
  }
}
