/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients;

import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

import com.mulesoft.anypoint.backoff.session.SessionMetadata;
import com.mulesoft.mule.runtime.gw.client.dto.PlatformContractAdapter;
import com.mulesoft.mule.runtime.gw.client.exception.NormalizedExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.model.ApiClientsResponse;
import com.mulesoft.mule.runtime.gw.client.session.ApiPlatformSession;
import com.mulesoft.mule.runtime.gw.client.session.factory.ApiPlatformSessionFactory;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.TrackingInfo;
import com.mulesoft.mule.runtime.gw.notification.ApiContractsListener;

import java.util.List;

import org.slf4j.Logger;

public class PlatformClientsRetriever implements ApiContractsListener {

  private static final String OBTAIN_CONTRACT_DESCRIPTION = "obtain contracts for API";

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(PlatformClientsRetriever.class);

  private final NormalizedExceptionMessageLogger normalizedLogger = new RecoverableExceptionMessageLogger(LOGGER);
  private final ApiPlatformSessionFactory platformSessionFactory;
  private final ApiTrackingService apiTrackingService;

  public PlatformClientsRetriever(ApiPlatformSessionFactory platformSessionFactory, ApiTrackingService apiTrackingService) {
    this.platformSessionFactory = platformSessionFactory;
    this.apiTrackingService = apiTrackingService;
  }


  public SessionMetadata retrieve(List<Api> apis) {
    ApiPlatformSession platform = platformSessionFactory.create();

    apis.forEach(api -> retrieve(platform, api));

    return platform.metadata();
  }

  @Override
  public void onContractsRequired(Api api) {
    ApiPlatformSession platform;
    try {
      platform = platformSessionFactory.create();
    } catch (Throwable e) {
      normalizedLogger.warn(OBTAIN_CONTRACT_DESCRIPTION, api, e);
      return;
    }

    try {
      retrieve(platform, api);
    } catch (Throwable e) {
      // Already logged.
    }
  }

  protected void retrieve(ApiPlatformSession platform, Api api) {

    LOGGER.debug("Fetching clients for {}", api);

    try {
      TrackingInfo trackingInfo = api.getTrackingInfo();
      ApiClientsResponse apiClientsResponse =
          platform.getApiClients(trackingInfo.getOrganizationId(), trackingInfo.getEnvironmentId(),
                                 api.getKey().id(), trackingInfo.getContractsEntityTag());
      if (apiClientsResponse.hasUpdates()) {
        List<PlatformContractAdapter> clients =
            apiClientsResponse.getClients()
                .stream()
                .map(PlatformContractAdapter::new)
                .filter(this::withNullIdOrSecret)
                .collect(toList());

        LOGGER.debug("Retrieved {} clients for {}", clients.size(), api);

        apiTrackingService.clientsPolling(api.getKey(), clients, apiClientsResponse.getContractsEntityTag());
      }
    } catch (Exception e) {
      normalizedLogger.warn(OBTAIN_CONTRACT_DESCRIPTION, api, e);
    } catch (Throwable e) {
      LOGGER.error(format("Unexpected error occurred while retrieving clients for API %s. %s", api, errorMessage(e)), e);
      throw e;
    }
  }

  private boolean withNullIdOrSecret(PlatformContractAdapter contract) {
    if (contract.clientId() == null) {
      LOGGER.trace("Null clientId found for client");

      return false;
    } else if (contract.clientSecret() == null) {
      LOGGER.trace("Null secret found for clientId");

      return false;
    }
    return true;
  }

}
