/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.hdp.registry;

import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getApiRegistryFolder;
import static org.apache.commons.io.filefilter.FileFilterUtils.makeFileOnly;
import static org.apache.commons.io.filefilter.FileFilterUtils.nameFileFilter;

import org.mule.runtime.api.lifecycle.Lifecycle;

import com.mulesoft.mule.runtime.gw.deployment.ApiService;

import java.io.File;

import org.apache.commons.io.monitor.FileAlterationListener;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Monitors the registration/deregistration of APIs for a high density proxy
 */
public class ApiRegistryWatcher implements Lifecycle {

  private static final Logger LOGGER = LoggerFactory.getLogger(ApiRegistryWatcher.class);

  private static final long POLLING_INTERVAL = 100;

  private final FileAlterationMonitor monitor;
  private final FileAlterationListener listener;
  private final String artifactName;

  public ApiRegistryWatcher(ApiService apiService, String artifactName) {
    this.artifactName = artifactName;
    this.monitor = new FileAlterationMonitor(POLLING_INTERVAL);
    this.listener = new ApiRegistryListener(apiService, artifactName);
  }

  @Override
  public void initialise() {
    FileAlterationObserver observer =
        new FileAlterationObserver(getApiRegistryFolder(), makeFileOnly(nameFileFilter(getFileName())));
    observer.addListener(listener);

    monitor.addObserver(observer);
    monitor.setThreadFactory(r -> {
      Thread t = new Thread(r);
      t.setName(String.format("agw-hdp-%s-watcher", artifactName));
      return t;
    });
  }

  @Override
  public void start() {
    try {
      File registryFile = new File(getApiRegistryFolder(), getFileName());
      LOGGER.debug("High Density Proxy API registry monitor started at {}",
                   registryFile.getCanonicalPath());
      monitor.start();

      // initial api registry load
      if (registryFile.isFile()) {
        listener.onFileCreate(registryFile);
      }
    } catch (Exception e) {
      LOGGER.error("Error starting High Density Proxy API registry monitor.", e);
    }
  }

  @Override
  public void stop() {
    try {
      LOGGER.debug("High Density Proxy API registry monitor stopped for application {}" + artifactName);
      monitor.stop();
    } catch (Exception e) {
      LOGGER.error("Error stopping High Density Proxy API registry monitor for application " + artifactName, e);
    }
  }

  @Override
  public void dispose() {}

  public String getArtifactName() {
    return artifactName;
  }

  private String getFileName() {
    return artifactName + ".json";
  }

}
