/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.metrics;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.concurrent.Executors.newScheduledThreadPool;

import org.mule.runtime.core.api.util.concurrent.NamedThreadFactory;
import org.mule.runtime.module.deployment.api.DeploymentService;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.metrics.event.Event;
import com.mulesoft.mule.runtime.gw.metrics.event.status.ApiRequestsTracker;
import com.mulesoft.mule.runtime.gw.metrics.processor.QueueEventProcessor;
import com.mulesoft.mule.runtime.gw.metrics.sender.MetricsScheduler;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentTracker;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueue;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueueFactory;

import java.util.ArrayDeque;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GatewayMetricsFactory {

  private static final Logger LOGGER = LoggerFactory.getLogger(GatewayMetricsFactory.class);
  public static final String AGW_METRICS_COLLECTOR_THREAD_NAME = "agw-metrics-collector";
  private static final int TO_SECONDS = 60;

  public Optional<GatewayMetricsAdapter> from(ApiPlatformClient apiPlatformClient,
                                              GatewayConfiguration configuration,
                                              DeploymentService deploymentService,
                                              ApiService apiService,
                                              PolicyDeploymentTracker policyTracker) {
    if (!configuration.metrics().metricsEnabled() || configuration.platformClient().isOfflineModeEnabled()) {
      LOGGER.debug("Gateway metrics module is disabled, no gateway usage metrics will be generated.");
      return empty();
    }

    String uuid = UUID.randomUUID().toString();

    SizeLimitedQueue<Event> queue = queue(configuration.metrics().getMetricsCapacity());

    //This thread won't do failing operations, therefore we don't need backoff nor retry
    ScheduledExecutorService executorService =
        newScheduledThreadPool(1, new NamedThreadFactory(AGW_METRICS_COLLECTOR_THREAD_NAME));

    int statusFrequency = configuration.metrics().getMetricsStatusFrequency();
    VariableRateExecutorService variableExecutor =
        new VariableRateExecutorService(executorService, startingDelay(statusFrequency));

    ApiRequestsTracker requestsTracker = new ApiRequestsTracker(apiService);

    GatewayMetricsAdapter adapter = new GatewayMetricsAdapter(
                                                              new GatewayMetrics(new QueueEventProcessor(queue)),
                                                              deploymentService,
                                                              apiService,
                                                              policyTracker,
                                                              variableExecutor, // Pass as a parameter for disposal and rate changing
                                                              requestsTracker,
                                                              new MetricsScheduler(uuid, queue, apiPlatformClient,
                                                                                   executorService));

    adapter.generateStatusAtRate(statusFrequency);
    policyTracker.addPolicyDeploymentListener(adapter);
    apiService.addDeploymentListener(adapter);
    apiService.addDeploymentListener(requestsTracker);

    return of(adapter);
  }

  /**
   * randomizes the starting delay to avoid all gateways going to the backend at the same time.
   *
   * @param maximum value for the initial delay
   * @return the delay that will be used for the initial status generation.
   */
  private int startingDelay(int maximum) {
    int min = maximum / 10;
    int max = maximum - min;
    return min + new Random().nextInt(max);
  }

  SizeLimitedQueue<Event> queue(int queueCapacity) {
    // TODO AGW-4933: Usage of ArrayBlocking Queue provides thread safe for the underlying queue but might have an impact on
    // performance
    return new SizeLimitedQueueFactory()
        .createConcurrentCircularQueue(new ArrayDeque<>(queueCapacity), queueCapacity);
  }
}
