/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.metrics.sender;

import com.mulesoft.mule.runtime.gw.backoff.configuration.BackoffConfiguration;
import com.mulesoft.mule.runtime.gw.backoff.scheduler.runnable.BackoffRunnable;
import com.mulesoft.mule.runtime.gw.backoff.session.BackoffBarrier;
import com.mulesoft.mule.runtime.gw.backoff.session.SessionMetadata;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.session.metadata.ApiPlatformSessionMetadata;
import com.mulesoft.mule.runtime.gw.metrics.event.Event;
import com.mulesoft.mule.runtime.gw.metrics.serialization.EventMapper;
import com.mulesoft.mule.runtime.gw.queue.SizeLimitedQueue;
import com.mulesoft.mule.runtime.gw.retry.session.ErrorSessionMetadata;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetricsSenderRunnable extends BackoffRunnable {

  private static final Logger LOGGER = LoggerFactory.getLogger(MetricsSenderRunnable.class);

  private final EventMapper eventMapper;
  private final SizeLimitedQueue<Event> senderEventQueue;
  private final ApiPlatformClient apiPlatformClient;
  private BackoffConfiguration backoffConfiguration;

  public MetricsSenderRunnable(EventMapper eventMapper,
                               SizeLimitedQueue<Event> senderEventQueue,
                               ApiPlatformClient apiPlatformClient,
                               BackoffConfiguration configuration,
                               BackoffBarrier backoffBarrier) {
    super(configuration, backoffBarrier);
    this.backoffConfiguration = configuration;
    this.eventMapper = eventMapper;
    this.senderEventQueue = senderEventQueue;
    this.apiPlatformClient = apiPlatformClient;
  }

  @Override
  protected SessionMetadata execute() {
    int statusCode = 200;

    List<Event> events = getEvents();

    try {
      if (!events.isEmpty()) {
        statusCode = apiPlatformClient.postMetrics(eventMapper.serialize(events));

        if (isFailureStatusCode(statusCode)) {
          preserveEvents(events);
        }
      }
    } catch (Throwable e) {
      LOGGER.debug("Unable to post metrics.", e);
      preserveEvents(events);
      return new ErrorSessionMetadata();
    }

    return new ApiPlatformSessionMetadata(statusCode);
  }

  private void preserveEvents(List<Event> events) {
    // The original order of events may change if we need to preserve the events. But as each event contains a timestamp,
    // the order in which we send the events to the ingest server is irrelevant.
    events.forEach(senderEventQueue::add);
  }

  private boolean isFailureStatusCode(int statusCode) {
    return backoffConfiguration.statusCodes().contains(statusCode);
  }

  private List<Event> getEvents() {
    List<Event> events = new ArrayList<>();

    Optional<Event> event = senderEventQueue.retrieve();

    while (event.isPresent()) {
      events.add(event.get());
      event = senderEventQueue.retrieve();
    }

    return events;
  }
}
