/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.metrics;

import static java.lang.Math.min;
import static java.util.Optional.ofNullable;

import org.mule.runtime.api.lifecycle.Disposable;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Wraps the executor service to reduce the complexity of switching execution rates.
 */
public class VariableRateExecutorService implements Disposable {

  private static final Logger LOGGER = LoggerFactory.getLogger(VariableRateExecutorService.class);

  private final ScheduledExecutorService executorService;
  private final int startingDelay;
  private final Map<String, ScheduledFuture<?>> scheduledTasks;

  public VariableRateExecutorService(ScheduledExecutorService executorService, int startingDelay) {
    this.executorService = executorService;
    this.startingDelay = startingDelay;
    this.scheduledTasks = new HashMap<>();
  }

  /**
   * Schedules at a periodic rate the incoming runnable. All tasks will be be delayed by min(startingDelay, rate)
   *
   * @param rate in seconds
   * @param runnable
   */
  public void atRate(String key, int rate, Runnable runnable) {
    ofNullable(scheduledTasks.remove(key))
        .ifPresent(this::cancelTask);

    doSchedule(key, rate, runnable);
  }

  @Override
  public void dispose() {
    scheduledTasks
        .forEach((key, future) -> cancelTask(future));

    executorService.shutdown();
  }

  private void cancelTask(ScheduledFuture<?> future) {
    try {
      if (!future.cancel(false)) {
        future.get(); // Wait for the task to end since it was currently executing
      }
    } catch (InterruptedException | ExecutionException e) {
      LOGGER.debug("Unexpected error while stopping the scheduled task ", e);
    }
  }

  private void doSchedule(String key, int rate, Runnable runnable) {
    if (rate > 0) {
      scheduledTasks.put(key, executorService.scheduleAtFixedRate(runnable, min(startingDelay, rate), rate, TimeUnit.SECONDS));
    }
  }

}
