/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.metrics.serialization;

import static java.util.stream.Collectors.toList;

import com.mulesoft.mule.runtime.gw.metrics.event.ApiDeployed;
import com.mulesoft.mule.runtime.gw.metrics.event.ApiUndeployed;
import com.mulesoft.mule.runtime.gw.metrics.event.Event;
import com.mulesoft.mule.runtime.gw.metrics.event.EventVisitor;
import com.mulesoft.mule.runtime.gw.metrics.event.PolicyApplied;
import com.mulesoft.mule.runtime.gw.metrics.event.PolicyRemoved;
import com.mulesoft.mule.runtime.gw.metrics.event.information.GatewayInformation;
import com.mulesoft.mule.runtime.gw.metrics.event.status.GatewayStatus;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.ApiDeployedDto;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.ApiUndeployedDto;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.EventDto;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.GatewayInformationDto;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.GatewayStatusDto;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.PolicyAppliedDto;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.PolicyRemovedDto;
import com.mulesoft.mule.runtime.gw.metrics.serialization.dto.EventMetadata;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;

import java.util.List;

public class EventMapper implements EventVisitor<EventDto> {

  private final EventMetadata metadata;
  private final ObjectWriter writer;

  public EventMapper(String uuid, String orgId) {
    writer = new ObjectMapper().writerFor(new TypeReference<List<EventDto>>() {});
    metadata = new EventMetadata(uuid, orgId);
  }

  public String serialize(List<Event> events) throws JsonProcessingException {
    List<EventDto> collect = events.stream().map(event -> event.accept(this)).collect(toList());
    return writer.writeValueAsString(collect);
  }

  @Override
  public EventDto visit(ApiDeployed apiDeployed) {
    return new ApiDeployedDto(metadata, apiDeployed);
  }

  @Override
  public EventDto visit(ApiUndeployed apiUndeployed) {
    return new ApiUndeployedDto(metadata, apiUndeployed);
  }

  @Override
  public EventDto visit(GatewayInformation gatewayInformation) {
    return new GatewayInformationDto(metadata, gatewayInformation);
  }

  @Override
  public EventDto visit(GatewayStatus gatewayStatus) {
    return new GatewayStatusDto(metadata, gatewayStatus);
  }

  @Override
  public EventDto visit(PolicyApplied policyApplied) {
    return new PolicyAppliedDto(metadata, policyApplied);
  }

  @Override
  public EventDto visit(PolicyRemoved policyRemoved) {
    return new PolicyRemovedDto(metadata, policyRemoved);
  }
}
