/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.metrics.event.status;

import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// TODO: AGW-4933: Review performance for this implementation
public class ApiRequestsTracker implements ApiDeploymentListener {

  private static final Logger LOGGER = LoggerFactory.getLogger(ApiRequestsTracker.class);
  private static final int INITIAL_MAP_CAPACITY = 1000;

  private ApiService apiService;
  private Map<Long, AtomicLong> requestsPerApi;

  /**
   * We use {@link ApiService} in order to keep requests track only of the APIs currently deployed in the Mule Gateway.
   *
   * @param apiService to be used to keep track of APIs
   */
  public ApiRequestsTracker(ApiService apiService) {
    this.apiService = apiService;
    this.requestsPerApi = newMap();
  }

  public void onApiDeploymentStart(Api api) {
    // Pre-initialized to avoid losing the requests received by the API on parallel hit initialization.
    requestsPerApi.put(api.getKey().id(), new AtomicLong());
  }

  /**
   * Thread safe operation to add a new request to the API request counter.
   *
   * @param apiId which received a request and need to add one to the counter.
   * @return this.
   */
  public ApiRequestsTracker addRequest(long apiId) {
    return addRequests(apiId, 1L);
  }

  ApiRequestsTracker addRequests(long apiId, long requests) {
    try {
      requestsPerApi.get(apiId).addAndGet(requests);
    } catch (Throwable t) {
      // We are using a plain hashmap to avoid the performance penalization, therefore when the hashmap
      // is resizing we risk not retrieving the correct value for the given key.
      LOGGER.trace("Suppressing unexpected error while counting tps for api {}.", apiId, t);
    }
    return this;
  }

  /**
   * Returns a snapshot of the requests received and resets counters for next {@link GatewayStatus} event.
   *
   * @return a {@link RequestsPerApi} containing the requests received at the moment of the method call.
   */
  public RequestsPerApi getRequestsAndReset() {
    // Replacing the pointer reference is an atomic operation in java, therefore we took advantage of it for creating the new map
    // which will be used during the following iteration of the periodic status event creation
    // Keeping all the ever deployed apis introduces the possibility of a memory leak. That's why we use the apiService to only
    // keep the currently deployed APIs.
    Map<Long, AtomicLong> oldRequestsPerApi = requestsPerApi;
    Map<Long, AtomicLong> newRequestsPerApi = newMap();
    apiService.getApis().forEach(api -> newRequestsPerApi.put(api.getKey().id(), new AtomicLong()));
    requestsPerApi = newRequestsPerApi;
    return new RequestsPerApi(oldRequestsPerApi);
  }

  private Map<Long, AtomicLong> newMap() {
    return new HashMap<>(INITIAL_MAP_CAPACITY);
  }

  @Override
  public String toString() {
    return requestsPerApi.toString();
  }

}
