/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.metrics;

import org.mule.runtime.api.lifecycle.Disposable;

import com.mulesoft.mule.runtime.gw.metrics.event.information.AnalyticsInformation;
import com.mulesoft.mule.runtime.gw.metrics.event.ApiDeployed;
import com.mulesoft.mule.runtime.gw.metrics.event.ApiUndeployed;
import com.mulesoft.mule.runtime.gw.metrics.event.Event;
import com.mulesoft.mule.runtime.gw.metrics.event.status.GatewayStatus;
import com.mulesoft.mule.runtime.gw.metrics.event.information.GatewayInformation;
import com.mulesoft.mule.runtime.gw.metrics.event.information.GatewaySecurityInformation;
import com.mulesoft.mule.runtime.gw.metrics.event.information.PlatformInformation;
import com.mulesoft.mule.runtime.gw.metrics.event.PolicyApplied;
import com.mulesoft.mule.runtime.gw.metrics.event.PolicyRemoved;
import com.mulesoft.mule.runtime.gw.metrics.event.status.AppStatus;
import com.mulesoft.mule.runtime.gw.metrics.processor.Processor;

import java.util.List;

import java.time.Clock;

/**
 * GatewayMetrics is the one in charge creates to create and forward the events to the queue. To ensure an easy backporting of the
 * implementation of metrics, the parameters of this class must be agnostic to all ApiGateway's classes and use only primitives.
 */
public class GatewayMetrics implements Disposable {

  private Processor<Event> processor;
  // AGW-4954 : Review clock usage (Perhaps Mule clock?)
  private Clock clock = Clock.systemDefaultZone();

  public GatewayMetrics(Processor<Event> processor) {
    this.processor = processor;
  }

  public void apiDeployed(long apiId, String apiName, String domainName) {
    submit(new ApiDeployed(apiId, apiName, domainName, currentMillis()));
  }

  public void apiUndeployed(long apiId) {
    submit(new ApiUndeployed(apiId, currentMillis()));
  }

  public void policyDeployed(Long apiId, String policyId, String gav, boolean online) {
    submit(new PolicyApplied(apiId, policyId, gav, online, currentMillis()));
  }

  public void policyRemoved(Long apiId, String policyId, String gav, boolean online) {
    submit(new PolicyRemoved(apiId, policyId, gav, online, currentMillis()));
  }

  public void gatewayInformation(String runtimeVersion, String deploymentTarget, String gatekeeperMode,
                                 boolean clusterMode, GatewaySecurityInformation gatewaySecurityInformation,
                                 PlatformInformation platformInformation, AnalyticsInformation analyticsInformation) {
    submit(new GatewayInformation(runtimeVersion, deploymentTarget, gatekeeperMode, clusterMode,
                                  gatewaySecurityInformation, platformInformation, analyticsInformation,
                                  currentMillis()));
  }

  public void status(List<AppStatus> apps) {
    submit(new GatewayStatus(apps, currentMillis()));
  }

  private long currentMillis() {
    return clock.instant().getEpochSecond();
  }

  public void submit(Event event) {
    processor.accept(event);
  }

  public void dispose() {
    processor.dispose();
  }
}
