/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.metrics;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentService;

import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.analytics.DeploymentType;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientConnectionListener;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.metrics.event.status.ApiRequestsTracker;
import com.mulesoft.mule.runtime.gw.metrics.event.status.RequestsPerApi;
import com.mulesoft.mule.runtime.gw.metrics.event.status.ApiStatus;
import com.mulesoft.mule.runtime.gw.metrics.event.status.AppStatus;
import com.mulesoft.mule.runtime.gw.metrics.event.status.PolicyStatus;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.metrics.event.information.AnalyticsInformation;
import com.mulesoft.mule.runtime.gw.metrics.event.information.GatewaySecurityInformation;
import com.mulesoft.mule.runtime.gw.metrics.event.information.PlatformInformation;
import com.mulesoft.mule.runtime.gw.metrics.sender.StatusListener;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentTracker;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * GatewayMetricsAdapter receives the parameters specific to the current Implementation and extracts the necessary information in
 * primitive types.
 *
 * All changes when backporting the metrics module should done in this class.
 */
public class GatewayMetricsAdapter
    implements PolicyDeploymentListener, ApiDeploymentListener, ApiPlatformClientConnectionListener, Disposable {

  private static final Logger LOGGER = LoggerFactory.getLogger(GatewayMetricsAdapter.class);

  private static final String STATUS_EVENT = "STATUS_EVENT";

  private GatewayMetrics gatewayMetrics;
  private DeploymentService deploymentService;
  private ApiService apiService;
  private PolicyDeploymentTracker policyTracker;
  private VariableRateExecutorService executorService;
  private ApiRequestsTracker apiRequestsTracker;
  private StatusListener statusListener;

  public GatewayMetricsAdapter(GatewayMetrics gatewayMetrics,
                               DeploymentService deploymentService,
                               ApiService apiService,
                               PolicyDeploymentTracker policyTracker,
                               VariableRateExecutorService executorService,
                               ApiRequestsTracker requestsTracker,
                               StatusListener statusListener) {
    this.gatewayMetrics = gatewayMetrics;
    this.deploymentService = deploymentService;
    this.apiService = apiService;
    this.policyTracker = policyTracker;
    this.executorService = executorService;
    this.apiRequestsTracker = requestsTracker;
    this.statusListener = statusListener;
  }

  @Override
  public void onClientConnected() {
    LOGGER.debug("Client connected. Starting scheduler.");
    try {
      statusListener.initialise();
    } catch (InitialisationException e) {
      LOGGER.debug("Unable to Start Metrics Scheduler after Platform Client Connected.", e);
    }
  }

  @Override
  public void onApiDeploymentStart(Api api) {
    LOGGER.debug("New Metrics event detected: Api Deployed");
    ApiImplementation apiImplementation = api.getImplementation();
    gatewayMetrics.apiDeployed(api.getKey().id(), apiImplementation.getArtifactName(), apiImplementation.getDomainName());
  }

  @Override
  public void onApiUndeploymentStart(ApiImplementation implementation) {
    LOGGER.debug("New Metrics event detected: Api Undeployed");
    gatewayMetrics.apiUndeployed(implementation.getApiKey().id());
  }

  @Override
  public void onApiRedeploymentStart(ApiImplementation implementation) {
    LOGGER.debug("New Metrics event detected: Api Undeployed");
    onApiUndeploymentStart(implementation);
  }

  @Override
  public void policyRemoved(ApiKey apiKey, PolicyDeploymentStatus status) {
    LOGGER.debug("New Metrics event detected: Policy removed");
    PolicyDefinitionDeploymentStatus policyDefinitionDeploymentStatus = status.getLatestPolicyStatus();
    if (policyDefinitionDeploymentStatus.isDeploymentSuccess()) {
      PolicyDefinition policy = policyDefinitionDeploymentStatus.getPolicyDefinition();
      gatewayMetrics.policyRemoved(apiKey.id(),
                                   policy.getId(),
                                   gav(policy.getTemplateKey()),
                                   policy.isOnline());
    }
  }

  @Override
  public void policyDeployed(ApiKey apiKey, PolicyDeploymentStatus status) {
    LOGGER.debug("New Metrics event detected: Policy deployed");
    PolicyDefinitionDeploymentStatus policyDefinitionDeploymentStatus = status.getLatestPolicyStatus();
    if (policyDefinitionDeploymentStatus.isDeploymentSuccess()) {
      PolicyDefinition policy = policyDefinitionDeploymentStatus.getPolicyDefinition();
      gatewayMetrics.policyDeployed(apiKey.id(),
                                    policy.getId(),
                                    gav(policy.getTemplateKey()),
                                    policy.isOnline());
    }
  }

  public void gatewayInformation(String runtimeVersion, DeploymentType deploymentType, boolean clusterMode,
                                 GatewayConfiguration gatewayConfiguration, AnalyticsConfiguration analyticsConfiguration) {
    LOGGER.debug("New Metrics event detected: Gateway initialization.");
    gatewayMetrics.gatewayInformation(runtimeVersion, deploymentType.toString(),
                                      gatewayConfiguration.gateKeeper().mode().toString(), clusterMode,
                                      gatewaySecurityInformation(gatewayConfiguration.securityConfiguration()),
                                      platformInformation(gatewayConfiguration.platformClient()),
                                      analyticsInformation(analyticsConfiguration));
  }

  /**
   * @param rate the seconds between successive event status generation. We are using seconds to enable short testing times.
   */
  public void generateStatusAtRate(int rate) {
    LOGGER.debug("Starting periodic event generation.");
    executorService.atRate(STATUS_EVENT, rate, this::generateStatusEvent);
  }

  public void generateStatusEvent() {
    LOGGER.debug("New Metrics event detected: Gateway status.");
    RequestsPerApi requestsPerApi = apiRequestsTracker.getRequestsAndReset();
    gatewayMetrics.status(deploymentService.getApplications().stream()
        .map(app -> extractAppInformation(app, requestsPerApi))
        .sorted()
        .collect(toList()));
  }

  public void requestReceived(AnalyticsHttpEvent analyticsHttpEvent) {
    apiRequestsTracker.addRequest(analyticsHttpEvent.getApiVersionId());
  }

  public void dispose() {
    gatewayMetrics.dispose();
    executorService.dispose();
    statusListener.dispose();
  }

  private AppStatus extractAppInformation(Application application, RequestsPerApi requestsPerApi) {
    String artifactName = application.getArtifactName();
    return new AppStatus(
                         artifactName,
                         apiService.getApis().stream()
                             .filter(api -> artifactName.equals(api.getImplementation().getArtifactName()))
                             .map(api -> extractApiInformation(api, requestsPerApi))
                             .sorted()
                             .collect(toList()));
  }

  private ApiStatus extractApiInformation(Api api, RequestsPerApi requestsPerApi) {
    long apiId = api.getKey().id();
    return new ApiStatus(apiId,
                         api.getImplementation().getDomainName(),
                         api.getTrackingInfo().isTracked(),
                         api.getTrackingInfo().getEndpointType(),
                         policyTracker.policyStatuses(api.getKey()).stream()
                             .filter(policyDeploymentStatus -> policyDeploymentStatus.getLatestPolicyStatus()
                                 .isDeploymentSuccess())
                             .map(this::extractPolicyInformation)
                             .sorted()
                             .collect(toList()),
                         requestsPerApi.requestsForApi(apiId));
  }

  private PolicyStatus extractPolicyInformation(PolicyDeploymentStatus status) {
    return new PolicyStatus(status.getLatestPolicyStatus().getPolicyDefinition().getOrder(),
                            status.getLatestPolicyStatus().getPolicyDefinition().getId(),
                            gav(status.getLatestPolicyStatus().getPolicyDefinition().getTemplateKey()),
                            status.getLatestPolicyStatus().getPolicyDefinition().isOnline());
  }

  private GatewaySecurityInformation gatewaySecurityInformation(GatewaySecurityConfiguration gatewaySecurityConfiguration) {
    return new GatewaySecurityInformation(gatewaySecurityConfiguration.hashClients(),
                                          gatewaySecurityConfiguration.isEncryptionEnabled(),
                                          gatewaySecurityConfiguration.isSensitiveOnlyEncryption());
  }

  private PlatformInformation platformInformation(PlatformClientConfiguration platformClientConfiguration) {
    return new PlatformInformation(platformClientConfiguration.getApisPollFrequency().inSeconds(),
                                   platformClientConfiguration.getClientsPollFrequency().inSeconds(),
                                   platformClientConfiguration.getApiKeepAliveFrequency().inSeconds(),
                                   platformClientConfiguration.getProxyHost() != null,
                                   platformClientConfiguration.backoffEnabled());
  }

  private AnalyticsInformation analyticsInformation(AnalyticsConfiguration analyticsConfiguration) {
    return new AnalyticsInformation(analyticsConfiguration.isEnabled(), analyticsConfiguration.getPushFrequency(),
                                    analyticsConfiguration.eventsThroughAgentEnabled());
  }

  private String gav(PolicyTemplateKey key) {
    return format("%s:%s:%s", key.getGroupId(), key.getAssetId(), key.getVersion());
  }
}
