/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.encryption.encrypter;

import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.lang.StringEscapeUtils.unescapeXml;

import com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypter;
import com.mulesoft.mule.runtime.gw.model.PolicyProperty;
import com.mulesoft.mule.runtime.gw.policies.encryption.EncryptedValueResult;

import java.util.List;
import java.util.Map;
import java.util.Optional;

public abstract class PolicyTypeEncrypter<T> {

  protected static final String TEMPLATE_PREFIX = "${secure::";
  protected static final String TEMPLATE_SUFFIX = "}";

  private final RuntimeEncrypter runtimeEncrypter;
  private final boolean isSensitiveOnlyEnabled;
  private final Map<String, PolicyProperty> policyProperties;

  public PolicyTypeEncrypter(RuntimeEncrypter runtimeEncrypter, boolean isSensitiveOnlyEnabled,
                             List<PolicyProperty> policyProperties) {
    this.runtimeEncrypter = runtimeEncrypter;
    this.isSensitiveOnlyEnabled = isSensitiveOnlyEnabled;
    this.policyProperties =
        policyProperties.stream().collect(toMap(PolicyProperty::getPropertyName, policyProperty -> policyProperty));
  }

  /**
   * Whether this encrypter supports treating the given key
   * 
   * @param key the key of the config data value that needs to be treated
   * @return true if it can be treated
   */
  public boolean supports(String key) {
    return supports(key, findProperty(key));
  }

  /**
   * Depending on configuration and value metadata, it encrypts or decrypts (if already encrypted) the given config data value
   * 
   * @param key the key of the config data value
   * @param value the actual value to encrypt or decrypt
   * @return an {@link EncryptedValueResult} containing the result of the process
   */
  public EncryptedValueResult encrypt(String key, T value) {
    Optional<PolicyProperty> policyProperty = findProperty(key);
    if (isSensitiveOnlyEnabled && policyProperty.isPresent() && !policyProperty.get().isSensitive()) {
      return decryptSingle(key, value);
    } else {
      return encryptSingle(key, value);
    }
  }

  /**
   * Encrypts the given value
   * 
   * @param key the key of the config data value
   * @param value the actual value to encrypt or decrypt
   * @return an {@link EncryptedValueResult} containing the result of the process
   */
  protected abstract EncryptedValueResult encryptSingle(String key, T value);

  /**
   * Decrypts the given value if it is encrypted. It does nothing to it if it is not
   * 
   * @param key the key of the config data value
   * @param value the actual value to encrypt or decrypt
   * @return an {@link EncryptedValueResult} containing the result of the process
   */
  protected abstract EncryptedValueResult decryptSingle(String key, T value);

  protected abstract boolean supports(String key, Optional<PolicyProperty> property);

  /**
   * Encrypts a single value.
   *
   * @param value The value that might be encrypted. It must be unescaped before encryption.
   *
   * @return The value encypted (that may be the same as value)
   */
  protected String encrypt(String value) {
    return runtimeEncrypter.encrypt(unescapeXml(value));
  }

  /**
   * Decrypts a single value.
   *
   * @param value The value that should be decrypted. This may be already encrypted or not. If it is not encrypted (the value is
   *              not between ![value] the decrypting function will leave the value untouched. In case of it being encrypted,
   *              it will try to decrypt it.
   *              In both cases, and same as before encryption, it should be unescaped to get the original value and avoid
   *              invalid characters when filling the template.
   *
   * @return The value decrypted (that may be the same as value)
   */
  protected String decrypt(String value) {
    return unescapeXml(runtimeEncrypter.decrypt(value));
  }

  private Optional<PolicyProperty> findProperty(String key) {
    return ofNullable(policyProperties.get(key));
  }
}
