/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.lifecyle;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatusTracker;
import com.mulesoft.mule.runtime.gw.notification.ApiContractsListener;
import com.mulesoft.mule.runtime.gw.notification.ApiDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;

import java.util.List;
import java.util.Optional;

/**
 * Monitors an api until policies and contracts are successfully applied then proceeds to unlock it.
 */
public abstract class GateKeeper
    implements PolicySetDeploymentListener, ApiContractsListener, ApiDeploymentListener {

  protected final ApiService apiService;

  public GateKeeper(ApiService apiService) {
    this.apiService = apiService;
  }

  /**
   * Check whether the API can be unblocked after a fresh {@link PolicySet} has been applied to it
   *
   * @param apiKey key of the API
   * @param policySet set of policies that were deployed
   * @param deploymentStatuses statuses of policies that were deployed
   */
  @Override
  public void onPolicySetDeploymentCompleted(ApiKey apiKey, PolicySet policySet,
                                             List<PolicyDeploymentStatus> deploymentStatuses) {

    getBlockedMonitoredApi(apiKey).ifPresent(api -> {

      GatekeeperStatusTracker status = api.getImplementation().gatekeeperStatus();

      if (validOrigin(policySet) && everyPolicySuccessfullyApplied(policySet.getPolicyDefinitions(), deploymentStatuses)) {
        status.policiesApplied();
      } else if (validOrigin(policySet)) {
        status.policyFailure();
      }

      if (status.shouldUnblock()) {
        unblockApi(api);
      }

    });
  }

  /**
   * Updates the gatekeeper status to indicate that contracts where correctly loaded.
   */
  @Override
  public void onContractsLoaded(ApiKey apiKey) {
    //When loading contracts from disk, this method can be called before the app is initially blocked.
    getMonitoredApi(apiKey).ifPresent(api -> {

      GatekeeperStatusTracker status = api.getImplementation().gatekeeperStatus();

      status.contractsAvailable();

      if (status.shouldUnblock()) {
        unblockApi(api);
      }
    });
  }

  @Override
  public void onContractsRequired(Api api) {
    if (isMonitored(api)) {
      api.getImplementation().gatekeeperStatus().contractsRequired();
    }
  }

  @Override
  public void onNoContractsRequired(Api api) {
    if (isMonitored(api)) {
      api.getImplementation().gatekeeperStatus().noContractsRequired();
    }
  }

  /**
   * @param apiKey the key of the API to retrieve.
   * @return the API corresponding to the apiKey if fulfills the requirements for the Gatekeeper to observe it, or Optional.empty() otherwise.
   */
  protected Optional<Api> getMonitoredApi(ApiKey apiKey) {
    return apiService.get(apiKey).filter(this::isMonitored);
  }

  /**
   * @param apiKey the key of the API to retrieve.
   * @return @return the API corresponding to the apiKey if it fulfills the requirements for the Gatekeeper to observe it has not yet been unblocked.
   */
  protected Optional<Api> getBlockedMonitoredApi(ApiKey apiKey) {
    return getMonitoredApi(apiKey)
        .filter(api -> api.getImplementation().gatekeeperStatus().isBlocked());
  }

  protected boolean everyPolicySuccessfullyApplied(List<PolicyDefinition> policyDefinitions,
                                                   List<PolicyDeploymentStatus> deploymentStatuses) {
    return policyDefinitions.stream().allMatch(policyDefinition -> deploymentStatuses.stream()
        .anyMatch(policyStatus -> policyDefinition.getName().equals(policyStatus.getPolicy().getPolicyDefinition().getName())
            && policyStatus.isDeploymentSuccess()));
  }

  /**
   * @param set PolicySet who's origin needs to be checked.
   * @return true if the origin of the policy set is valid for the gatekeeper configuration.
   */
  protected abstract boolean validOrigin(PolicySet set);

  protected abstract boolean isMonitored(Api api);

  protected abstract void blockApi(Api api);

  protected abstract void unblockApi(Api api);


  /**
   * Block API before listener is started
   *
   * @param api API implementation that it's deployment started
   */
  @Override
  public void onApiDeploymentStart(Api api) {
    if (isMonitored(api)) {
      blockApi(api);
    }
  }
}
