/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.serialization;

import static java.util.stream.Collectors.toList;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import com.fasterxml.jackson.annotation.JsonProperty;

import java.util.List;
import java.util.Map;

public class PolicyDefinitionDto {

  @JsonProperty("id")
  private String id;

  @JsonProperty("template")
  private PolicyTemplateKeyDTO templateKey;

  @JsonProperty("api")
  private List<APIKeyDTO> apis;

  @JsonProperty("online")
  private boolean online;

  @JsonProperty("order")
  private int order;

  @JsonProperty("pointcuts")
  private PointcutsDto pointcuts;

  @JsonProperty("configuration")
  private Map<String, Object> configuration;

  public static PolicyDefinitionDto from(PolicyDefinition policyDefinition) {
    PolicyDefinitionDto dto = new PolicyDefinitionDto();

    dto.id = policyDefinition.getId();
    dto.templateKey = new PolicyTemplateKeyDTO(policyDefinition.getTemplateKey());
    dto.apis = policyDefinition.getApiKeys().stream().map(APIKeyDTO::new).collect(toList());
    dto.online = policyDefinition.isOnline();
    dto.order = policyDefinition.getOrder();
    if (!policyDefinition.getHttpResourcePointcuts().isEmpty()) {
      dto.pointcuts = PointcutsDto.from(policyDefinition.getHttpResourcePointcuts());
    }
    dto.configuration = policyDefinition.getConfigurationData().getConfiguration();

    return dto;
  }

  public String getId() {
    return id;
  }

  public PolicyTemplateKeyDTO getTemplateKey() {
    return templateKey;
  }

  public int getOrder() {
    return order;
  }

  public Map<String, Object> getConfiguration() {
    return configuration;
  }

  public PointcutsDto getPointcuts() {
    return pointcuts;
  }

  public boolean getOnline() {
    return online;
  }

  public List<APIKeyDTO> getApis() {
    return apis;
  }

  static class APIKeyDTO {

    @JsonProperty("id")
    private Long id;

    public APIKeyDTO() {}

    public APIKeyDTO(ApiKey apiKey) {
      this.id = apiKey.id();
    }

    public Long getId() {
      return id;
    }

  }

  static class PolicyTemplateKeyDTO {

    @JsonProperty("groupId")
    private String groupId;

    @JsonProperty("assetId")
    private String assetId;

    @JsonProperty("version")
    private String version;

    public PolicyTemplateKeyDTO() {}

    public PolicyTemplateKeyDTO(PolicyTemplateKey key) {
      this.assetId = key.getAssetId();
      this.groupId = key.getGroupId();
      this.version = key.getVersion();
    }

    public String getGroupId() {
      return groupId;
    }

    public String getAssetId() {
      return assetId;
    }

    public String getVersion() {
      return version;
    }
  }
}
