/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service;

import static com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus.DEPLOYMENT_FAILED;
import static java.util.stream.Collectors.toList;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.deployment.DeploymentExceptionHandler;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.PolicySetDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;
import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateException;
import com.mulesoft.mule.runtime.gw.retry.RunnableRetrier;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArrayList;

import org.slf4j.Logger;

public class DefaultPolicySetDeploymentService implements PolicySetDeploymentService {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(DefaultPolicySetDeploymentService.class);

  private final RunnableRetrier<ApiKey> runnableRetrier;
  private final PolicyDeploymentService policyDeploymentService;
  private final PolicyDeploymentTracker policyDeploymentTracker;
  private final PolicyStore policyStore;
  private final List<PolicySetDeploymentListener> deploymentListeners;
  private final PolicyFactory policyFactory;
  private final ApiService apiService;

  public DefaultPolicySetDeploymentService(RunnableRetrier<ApiKey> policySetRetrier,
                                           PolicyDeploymentService policyDeploymentService,
                                           PolicyDeploymentTracker policyDeploymentTracker,
                                           PolicyStore policyStore,
                                           PolicyFactory policyFactory, ApiService apiService) {
    this.deploymentListeners = new CopyOnWriteArrayList<>();
    this.runnableRetrier = policySetRetrier;
    this.policyDeploymentService = policyDeploymentService;
    this.policyDeploymentTracker = policyDeploymentTracker;
    this.policyStore = policyStore;
    this.policyFactory = policyFactory;
    this.apiService = apiService;
  }

  /**
   * {@inheritDoc}
   * <p>
   * This implementation is synchronized to avoid a race condition in the case that this method is called a second time for the
   * same API when the processing of the first one has not finished. This is a real possibility since the callers are not
   * synchronized
   */
  @Override
  public synchronized void policiesForApi(ApiKey apiKey, PolicySet policySet) {
    runnableRetrier.scheduleRetry(apiKey, () -> {
      logPoliciesForApi(apiKey, policySet);
      updatePolicies(apiKey, policySet);

      if (failedDownloadTemplates(apiKey).size() == 0) {
        notifyPolicySetDeployed(apiKey, policySet);
      } else {
        LOGGER.debug("Template download failed for API {} - Policies {}.", apiKey, names(failedDownloadTemplates(apiKey)));
        throw new RuntimeException("There are still failures in template downloading");
      }
    });
  }

  @Override
  public void removeAll(ApiKey apiKey) {
    LOGGER.debug("Removing all policies from API {}", apiKey);

    // Notify listeners first, so they act before the API is not protected anymore
    notifyAllPoliciesRemoved(apiKey);

    List<Policy> policiesToRemove = policyDeploymentTracker.apiRemoved(apiKey);

    policiesToRemove.forEach(policyDeploymentService::removePolicy);
  }

  @Override
  public void onApiDeploymentSuccess(Api api) {
    // Deploy offline policies to API
    policyStore.offlinePolicies().forEach(policyDefinition -> policyDeploymentService
        .newPolicyForApi(policyFactory.createFromPolicyDefinition(policyDefinition), api.getKey()));
  }

  @Override
  public void onApiUndeploymentStart(ApiImplementation implementation) {
    removeAll(implementation.getApiKey());
  }

  @Override
  public void onApiRedeploymentStart(ApiImplementation implementation) {
    notifyAllPoliciesRemoved(implementation.getApiKey());
    policyDeploymentTracker.apiRemoved(implementation.getApiKey());
  }

  @Override
  public void addPolicyDeploymentListener(PolicySetDeploymentListener listener) {
    deploymentListeners.add(listener);
  }

  @Override
  public void conciliatePolicies(ApiKey apiKey, List<PolicyDefinition> desiredPolicies) {
    List<PolicyDefinition> actualPolicies = filterByApi(policyStore.onlinePolicies(), apiKey);

    actualPolicies.stream()
        .filter(policyDefinition -> !desiredPolicies.contains(policyDefinition))
        .forEach(policyDefinition -> policyStore.remove(policyDefinition.getName()));

    desiredPolicies.stream()
        .filter(policyDefinition -> !actualPolicies.contains(policyDefinition))
        .forEach(policyDefinition -> policyStore.store(policyDefinition)); // Template will be resolved when deployed
  }

  @Override
  public Map<ApiKey, List<PolicyDefinition>> storedOnlinePoliciesByApi() {
    Map<ApiKey, List<PolicyDefinition>> groupedPolicies = new HashMap<>();

    policyStore.onlinePolicies().forEach(policyDefinition -> policyDefinition.getApiKeys().forEach(apiKey -> {
      groupedPolicies.putIfAbsent(apiKey, new ArrayList<>());
      groupedPolicies.get(apiKey).add(policyDefinition);
    }));

    return groupedPolicies;
  }

  private void logPoliciesForApi(ApiKey apiKey, PolicySet policySet) {
    if (LOGGER.isDebugEnabled()) {
      List<String> policyNames = policySet.getPolicyDefinitions().stream().map(PolicyDefinition::getName).collect(toList());
      LOGGER
          .debug("Deploying policies {} from {} to API {}", policyNames, policySet.isFromPlatform() ? "Platform" : "File System",
                 apiKey);
    }
  }

  private void updatePolicies(ApiKey apiKey, PolicySet policySet) {
    List<PolicyDeploymentStatus> oldTrackingStatuses = policyDeploymentTracker.onlinePolicyStatuses(apiKey);

    for (PolicyDefinition policyDefinition : policySet.getPolicyDefinitions()) {
      Optional<PolicyDeploymentStatus> oldStatus = oldTrackingStatuses.stream()
          .filter(status -> samePolicyName(status, policyDefinition))
          .findFirst();

      try {
        Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

        if (!oldStatus.isPresent()) {
          LOGGER.debug("New policy {} detected to apply", policyDefinition.getName());
          policyDeploymentService.newPolicy(policy);
        } else if (!oldStatus.get().getPolicy().getPolicyDefinition().equals(policy.getPolicyDefinition())
            || oldStatus.get().isTemplateDownloadFailed()) {
          LOGGER.debug("Update on policy {} detected", policyDefinition.getName());
          policyDeploymentService.updatePolicy(policy);
        }
      } catch (PolicyTemplateException exception) {
        handleDeploymentException(policyDefinition, exception.status(), apiKey, exception);
      } catch (Exception exception) {
        handleDeploymentException(policyDefinition, DEPLOYMENT_FAILED, apiKey, exception);
      }

    }

    removeOldPolicies(policySet, oldTrackingStatuses);
  }

  private void notifyAllPoliciesRemoved(ApiKey apiKey) {
    deploymentListeners.forEach(policyDeploymentListener -> {
      try {
        policyDeploymentListener.onPoliciesRemoved(apiKey);
      } catch (Exception e) {
        LOGGER.warn("Error on polices removed listener: {}", e.getMessage());
      }
    });
  }

  private void removeOldPolicies(PolicySet policySet, List<PolicyDeploymentStatus> oldTrackingStatuses) {
    oldTrackingStatuses.stream()
        .filter(status -> policySet
            .getPolicyDefinitions().stream()
            .noneMatch(policyDefinition -> samePolicyName(status, policyDefinition)))
        .forEach(status -> {
          LOGGER.debug("Policy {} no longer present, it will be removed", status.getPolicy().getPolicyDefinition().getName());
          policyDeploymentService.removePolicy(status.getPolicy().getPolicyDefinition().getName());
        });
  }

  private void notifyPolicySetDeployed(ApiKey apiKey, PolicySet policySet) {
    deploymentListeners.forEach(policyDeploymentListener -> {
      try {
        policyDeploymentListener
            .onPolicySetDeploymentCompleted(apiKey, policySet, policyDeploymentTracker.onlinePolicyStatuses(apiKey));
      } catch (Exception e) {
        LOGGER.warn("Error on policy deployment completed listener: {}", e.getMessage());
      }
    });
  }

  private boolean samePolicyName(PolicyDeploymentStatus status, PolicyDefinition policyDefinition) {
    return policyDefinition.getName().equals(status.getPolicy().getPolicyDefinition().getName());
  }


  private List<PolicyDefinition> filterByApi(List<PolicyDefinition> policyDefinitions, ApiKey apiKey) {
    return policyDefinitions.stream()
        .filter(policyDefinition -> policyDefinition.getApiKeys().contains(apiKey))
        .collect(toList());
  }

  private List<String> names(List<PolicyDeploymentStatus> policyDeploymentStatuses) {
    return policyDeploymentStatuses.stream().map(status -> status.getPolicy().getPolicyDefinition().getName()).collect(toList());
  }

  private List<PolicyDeploymentStatus> failedDownloadTemplates(ApiKey apiKey) {
    return policyDeploymentTracker
        .onlinePolicyStatuses(apiKey).stream()
        .filter(PolicyDeploymentStatus::isTemplateDownloadFailed)
        .collect(toList());
  }

  private void handleDeploymentException(PolicyDefinition policyDefinition,
                                         DeploymentStatus deploymentStatus, ApiKey apiKey,
                                         Exception exception) {
    // If policy couldn't be resolved, store it anyway, since it might have been fixed for the next restart
    policyStore.store(policyDefinition);
    PolicyDeploymentStatus status = new PolicyDeploymentStatus(new Policy(null, policyDefinition, null), deploymentStatus);
    DeploymentExceptionHandler deploymentExceptionHandler = new DeploymentExceptionHandler(policyStore);
    apiService.get(apiKey)
        .ifPresent(api -> deploymentExceptionHandler.handle(policyDefinition, api.getImplementation(), exception));
    policyDeploymentTracker.policyDeployed(apiKey, status);
  }

}
