/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.encryption;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypterFactory.createDefaultRuntimeEncrypter;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypter;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicyProperty;
import com.mulesoft.mule.runtime.gw.model.PolicySpecification;

import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class SensitivePolicyConfigurationEncrypterTestCase extends AbstractMuleTestCase {

  private static final String KEY = "Key";
  private static final String KEY_2 = "Key2";
  private static final String KEY_LIST_STRING = "List_String";
  private static final String KEY_BOOLEAN = "Key_Boolean";
  private static final String KEY_KEYVALUE = "Keyvalue";
  private static final String KEY_LIST_KEYVALUES = "List_Keyvalues";
  private static final String KEY_NOT_DEFINED = "KeyNotDefinedInYaml";

  private static final String ENCRYPTION_KEY = "GatewayTeamKey00";

  private static final String ENCRYPTED_VALUE = "![z8p1poC0yyjUoTNzZhS2Xw==]";
  private static final String ENCRYPTED_VALUE_2 = "![dz3YF4DaHoJNwbdLD8Uzxg==]";
  private static final String ENCRYPTED_TEMPLATE_PLACEHOLDER = "${secure::%s}";
  private static final String SIMPLE_KEY = "mapKey";
  private static final String SIMPLE_KEY_2 = "mapKey2";
  private static final String SIMPLE_VALUE = "Test Value";
  private static final String SIMPLE_VALUE_2 = "Test Value2";
  private static final String HTML_ENCODED_EXPRESSION = "#[attributes.headers[&#39;client_secret&#39;]]";
  private static final String EXPRESSION = "#[attributes.headers['client_secret']]";

  @Rule
  public ExpectedException expectedException = none();

  @Rule
  public SystemProperty encryptionKey = new SystemProperty("anypoint.platform.encryption_key", ENCRYPTION_KEY);

  private PolicyConfigurationEncrypter sensitiveOnlyEncrypter;

  @Before
  public void setUp() {
    RuntimeEncrypter runtimeEncrypter = createDefaultRuntimeEncrypter();
    sensitiveOnlyEncrypter = new DefaultPolicyConfigurationEncrypter(runtimeEncrypter, true);
  }

  @Test
  public void encryptString() {
    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY, SIMPLE_VALUE)), sensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY), is(ENCRYPTED_VALUE));
    assertThat(result.getConfigFileProperties().get(KEY), is(ENCRYPTED_VALUE));
    assertThat(result.getTemplatePlaceholders().get(KEY), is(encryptedTemplatePlaceholder(KEY)));
  }

  @Test
  public void decryptingExpressionWithHTMLEncodedCharacters() {
    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY, HTML_ENCODED_EXPRESSION)),
                                                    nonSensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));

    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY), is(EXPRESSION));
    assertThat(result.getTemplatePlaceholders().get(KEY), is(EXPRESSION));
    assertThat(result.getConfigFileProperties().isEmpty(), is(true));
  }

  @Test
  public void sensitiveOnlyEnabledDoesDecryptsNonSensitiveString() {
    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY, ENCRYPTED_VALUE,
                                                                        KEY_2, SIMPLE_VALUE_2)),
                                                    nonSensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY), is(SIMPLE_VALUE));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY_2), is(SIMPLE_VALUE_2));
    assertThat(result.getTemplatePlaceholders().get(KEY), is(SIMPLE_VALUE));
    assertThat(result.getTemplatePlaceholders().get(KEY_2), is(SIMPLE_VALUE_2));
    assertThat(result.getConfigFileProperties().isEmpty(), is(true));
  }

  @Test
  public void encryptBoolean() {
    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY_BOOLEAN, true)), sensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY_BOOLEAN), is(true));
    assertThat(result.getConfigFileProperties().get(KEY_BOOLEAN), nullValue());
    assertThat(result.getTemplatePlaceholders().get(KEY_BOOLEAN), is(true));
  }

  @Test
  public void encryptNotDefinedKey() {
    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY_NOT_DEFINED, SIMPLE_VALUE)),
                                                    sensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY_NOT_DEFINED), is(SIMPLE_VALUE));
    assertThat(result.getConfigFileProperties().get(KEY_NOT_DEFINED), nullValue());
    assertThat(result.getTemplatePlaceholders().get(KEY_NOT_DEFINED), is(SIMPLE_VALUE));
  }

  @Test
  public void encryptListOfStrings() {
    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY_LIST_STRING, asList(SIMPLE_VALUE, SIMPLE_VALUE_2))),
                                                    sensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertThat((List<String>) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_STRING),
               contains(ENCRYPTED_VALUE, ENCRYPTED_VALUE_2));
    assertThat((List<String>) result.getTemplatePlaceholders().get(KEY_LIST_STRING),
               contains(encryptedTemplatePlaceholder(KEY_LIST_STRING + ".0"),
                        encryptedTemplatePlaceholder(KEY_LIST_STRING + ".1")));
    assertThat(((Map) result.getConfigFileProperties().get(KEY_LIST_STRING)).get("0"), is(ENCRYPTED_VALUE));
    assertThat(((Map) result.getConfigFileProperties().get(KEY_LIST_STRING)).get("1"), is(ENCRYPTED_VALUE_2));
  }

  @Test
  public void sensitiveOnlyEnabledDecryptsNonSensitiveListOfString() {
    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter
            .encryptConfiguration(policyDefinition(of(KEY_LIST_STRING, asList(ENCRYPTED_VALUE, SIMPLE_VALUE_2))),
                                  nonSensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertThat((List<String>) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_STRING),
               contains(SIMPLE_VALUE, SIMPLE_VALUE_2));
    assertThat((List<String>) result.getTemplatePlaceholders().get(KEY_LIST_STRING),
               contains(SIMPLE_VALUE, SIMPLE_VALUE_2));
    assertThat((result.getConfigFileProperties().get(KEY_LIST_STRING)), nullValue());
  }

  @Test
  public void encryptKeyvalue() {
    Map<String, String> propertyMap = of("key", SIMPLE_KEY, "value", SIMPLE_VALUE);

    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY_KEYVALUE, propertyMap)),
                                                    sensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertKeyValue((Map<String, Object>) result.getConfigurationProperties().getConfiguration().get(KEY_KEYVALUE), SIMPLE_KEY,
                   ENCRYPTED_VALUE);
    assertKeyValue((Map) result.getTemplatePlaceholders().get(KEY_KEYVALUE), SIMPLE_KEY,
                   encryptedTemplatePlaceholder(KEY_KEYVALUE, SIMPLE_KEY));
    assertKeyValue((Map<String, Object>) result.getConfigFileProperties().get(KEY_KEYVALUE), SIMPLE_KEY,
                   ENCRYPTED_VALUE);
  }

  @Test
  public void decryptKeyvalue() {
    Map<String, String> propertyMap = of("key", SIMPLE_KEY, "value", ENCRYPTED_VALUE);

    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter.encryptConfiguration(policyDefinition(of(KEY_KEYVALUE, propertyMap)),
                                                    nonSensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertKeyValue((Map<String, Object>) result.getConfigurationProperties().getConfiguration().get(KEY_KEYVALUE), SIMPLE_KEY,
                   SIMPLE_VALUE);
    assertKeyValue((Map) result.getTemplatePlaceholders().get(KEY_KEYVALUE), SIMPLE_KEY, SIMPLE_VALUE);
    assertThat(result.getConfigFileProperties().isEmpty(), is(true));
  }

  @Test
  public void encryptListOfKeyvalues() {
    Map<String, String> propertyMap = of("key", SIMPLE_KEY, "value", SIMPLE_VALUE);
    Map<String, String> propertyMap2 = of("key", SIMPLE_KEY_2, "value", SIMPLE_VALUE_2);

    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter
            .encryptConfiguration(policyDefinition(of(KEY_LIST_KEYVALUES, asList(propertyMap, propertyMap2))),
                                  sensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertKeyValue((Map<String, Object>) ((List) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_KEYVALUES))
        .get(0), SIMPLE_KEY_2, ENCRYPTED_VALUE_2);
    assertKeyValue((Map<String, Object>) ((List) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_KEYVALUES))
        .get(1), SIMPLE_KEY, ENCRYPTED_VALUE);
    assertKeyValue((Map<String, Object>) ((List) result.getTemplatePlaceholders().get(KEY_LIST_KEYVALUES))
        .get(0), SIMPLE_KEY_2, encryptedTemplatePlaceholder(KEY_LIST_KEYVALUES, SIMPLE_KEY_2));
    assertKeyValue((Map<String, Object>) ((List) result.getTemplatePlaceholders().get(KEY_LIST_KEYVALUES))
        .get(1), SIMPLE_KEY, encryptedTemplatePlaceholder(KEY_LIST_KEYVALUES, SIMPLE_KEY));
    assertThat(((Map<String, Object>) result.getConfigFileProperties().get(KEY_LIST_KEYVALUES)).get(SIMPLE_KEY),
               is(ENCRYPTED_VALUE));
    assertThat(((Map<String, Object>) result.getConfigFileProperties().get(KEY_LIST_KEYVALUES)).get(SIMPLE_KEY_2),
               is(ENCRYPTED_VALUE_2));
  }

  @Test
  public void decryptListOfKeyValues() {
    Map<String, String> propertyMap = of("key", SIMPLE_KEY, "value", ENCRYPTED_VALUE);
    Map<String, String> propertyMap2 = of("key", SIMPLE_KEY_2, "value", SIMPLE_VALUE_2);

    PolicyConfigurationEncryptionResult result =
        sensitiveOnlyEncrypter
            .encryptConfiguration(policyDefinition(of(KEY_LIST_KEYVALUES, asList(propertyMap, propertyMap2))),
                                  nonSensitiveSpecification());

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertKeyValue((Map<String, Object>) ((List) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_KEYVALUES))
        .get(0), SIMPLE_KEY, SIMPLE_VALUE);
    assertKeyValue((Map<String, Object>) ((List) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_KEYVALUES))
        .get(1), SIMPLE_KEY_2, SIMPLE_VALUE_2);
    assertKeyValue((Map<String, Object>) ((List) result.getTemplatePlaceholders().get(KEY_LIST_KEYVALUES))
        .get(0), SIMPLE_KEY, SIMPLE_VALUE);
    assertKeyValue((Map<String, Object>) ((List) result.getTemplatePlaceholders().get(KEY_LIST_KEYVALUES))
        .get(1), SIMPLE_KEY_2, SIMPLE_VALUE_2);
    assertThat(result.getConfigFileProperties().isEmpty(), is(true));
  }

  private PolicyDefinition policyDefinition(Map<String, Object> configData) {
    PolicyDefinition policyDefinition = mock(PolicyDefinition.class);

    when(policyDefinition.getConfigurationData()).thenReturn(new PolicyConfiguration(configData));

    return policyDefinition;
  }

  private PolicySpecification sensitiveSpecification() {
    return mockSpecification(true);
  }

  private PolicySpecification nonSensitiveSpecification() {
    return mockSpecification(false);
  }

  private PolicySpecification mockSpecification(boolean sensitive) {
    PolicySpecification policySpecification = mock(PolicySpecification.class);

    when(policySpecification.getConfiguration()).thenReturn(buildMockedPolicyProperties(sensitive));
    when(policySpecification.isValid()).thenReturn(true);
    return policySpecification;
  }

  private List<PolicyProperty> buildMockedPolicyProperties(boolean sensitive) {
    PolicyProperty policyProperty1 = new PolicyProperty();
    policyProperty1.setPropertyName(KEY);
    policyProperty1.setSensitive(sensitive);
    policyProperty1.setType("string");
    policyProperty1.setAllowMultiple(false);

    PolicyProperty policyProperty3 = new PolicyProperty();
    policyProperty3.setPropertyName(KEY_BOOLEAN);
    policyProperty3.setSensitive(sensitive);
    policyProperty3.setType("boolean");
    policyProperty3.setAllowMultiple(false);

    PolicyProperty policyProperty4 = new PolicyProperty();
    policyProperty4.setPropertyName(KEY_KEYVALUE);
    policyProperty4.setSensitive(sensitive);
    policyProperty4.setType("keyvalues");
    policyProperty4.setAllowMultiple(false);

    PolicyProperty policyProperty5 = new PolicyProperty();
    policyProperty5.setPropertyName(KEY_LIST_KEYVALUES);
    policyProperty5.setSensitive(sensitive);
    policyProperty5.setType("keyvalues");
    policyProperty5.setAllowMultiple(true);

    PolicyProperty policyProperty6 = new PolicyProperty();
    policyProperty6.setPropertyName(KEY_LIST_STRING);
    policyProperty6.setSensitive(sensitive);
    policyProperty6.setType("string");
    policyProperty6.setAllowMultiple(true);

    return asList(policyProperty1, policyProperty3, policyProperty4, policyProperty5, policyProperty6);
  }

  private void assertKeyValue(Map<String, Object> keyvalue, String key, String value) {
    assertThat(keyvalue.get("key"), is(key));
    assertThat(keyvalue.get("value"), is(value));
  }

  private String encryptedTemplatePlaceholder(String property) {
    return format(ENCRYPTED_TEMPLATE_PLACEHOLDER, property);
  }

  private String encryptedTemplatePlaceholder(String property, String innerProperty) {
    return format(ENCRYPTED_TEMPLATE_PLACEHOLDER, property + "." + innerProperty);
  }
}
