/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.factory;

import static com.google.common.collect.ImmutableMap.of;
import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Matchers.isNull;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.model.EmptyPolicySpecification;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySpecification;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.encryption.EncryptedPolicyException;
import com.mulesoft.mule.runtime.gw.policies.serialization.PolicySpecificationBuilder;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;
import com.mulesoft.mule.runtime.gw.policies.template.provider.PolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.resolver.PolicyTemplateResolver;

import java.io.File;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Matchers;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class DefaultPolicyFactoryTestCase extends AbstractMuleTestCase {

  private PolicyFactory policyFactory;

  @Mock
  private PolicyTemplate template;
  private String resolvedTemplate = "someTemplate";

  @Before
  public void setUp() {
    PolicyTemplateResolver templateResolver = mock(PolicyTemplateResolver.class);
    PolicyTemplateProvider templateProvider = mock(PolicyTemplateProvider.class);

    when(templateProvider.provide(any())).thenReturn(template);
    when(templateResolver.resolve(eq(template), any())).thenReturn(resolvedTemplate);

    when(template.getPolicySpecification()).thenReturn(new EmptyPolicySpecification());
    policyFactory = new DefaultPolicyFactory(templateResolver, templateProvider);
  }

  @Test
  public void createPolicy() {
    PolicyDefinition policyDefinition = mock(PolicyDefinition.class, RETURNS_DEEP_STUBS);

    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition(), is(policyDefinition));
    assertThat(policy.getResolvedTemplate(), is(resolvedTemplate));
  }

  @Test(expected = EncryptedPolicyException.class)
  public void encryptedPolicyWithNoKey() {
    PolicyDefinition policyDefinition = mock(PolicyDefinition.class, RETURNS_DEEP_STUBS);
    when(policyDefinition.getConfigurationData().isEncrypted()).thenReturn(true);

    policyFactory.createFromPolicyDefinition(policyDefinition);
  }

  @Test
  public void createPolicyRemovingIdentityManagement() {
    PolicyConfiguration policyConfiguration = new PolicyConfiguration(
                                                                      of("string", "value", "int", 60000,
                                                                         "identityManagementTokenUrl",
                                                                         "identityManagementTokenUrlValue",
                                                                         "identityManagementClientSecret",
                                                                         "u6gMLNLHDv4QRAqqzRezPqcDoohC0vMTvtKUqy5",
                                                                         "identityManagementClientId", "api_platform"));
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, new ApiKey(1L), newArrayList(new HttpResourcePointcut(".*", "GET")),
                             1, policyConfiguration);

    PolicySpecification policySpecification = new PolicySpecificationBuilder()
        .withFile(new File(getClass().getResource("/specs/client-id-enforcement.yaml").getFile()))
        .build();

    when(template.getPolicySpecification()).thenReturn(policySpecification);

    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration().get("identityManagementTokenUrl"),
               nullValue());
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration().get("identityManagementClientSecret"),
               nullValue());
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration().get("identityManagementClientId"),
               nullValue());
    assertThat(policy.getPolicyDefinition().getName(), is(policyDefinition.getName()));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(policyDefinition.getTemplateKey()));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), is(policyDefinition.getHttpResourcePointcuts()));
    assertThat(policy.getPolicyDefinition().getApiKeys(), is(policyDefinition.getApiKeys()));
    assertThat(policy.getPolicyDefinition().getOrder(), is(policyDefinition.getOrder()));
    assertThat(policy.getPolicyDefinition().getId(), is(policyDefinition.getId()));
    assertThat(policy.getResolvedTemplate(), is(resolvedTemplate));
  }

  @Test
  public void createPolicyNotRemovingIdentityManagement() {
    PolicyConfiguration policyConfiguration = new PolicyConfiguration(
                                                                      of("string", "value", "int", 60000,
                                                                         "identityManagementTokenUrl",
                                                                         "identityManagementTokenUrlValue",
                                                                         "identityManagementClientSecret",
                                                                         "u6gMLNLHDv4QRAqqzRezPqcDoohC0vMTvtKUqy5",
                                                                         "identityManagementClientId", "api_platform"));
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, new ApiKey(1L), newArrayList(new HttpResourcePointcut(".*", "GET")),
                             1, policyConfiguration);

    PolicySpecification policySpecification = new PolicySpecificationBuilder()
        .withFile(new File(getClass().getResource("/specs/openidconnect-access-token-enforcement.yaml").getFile()))
        .build();

    when(template.getPolicySpecification()).thenReturn(policySpecification);

    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration().get("identityManagementTokenUrl"),
               is("identityManagementTokenUrlValue"));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration().get("identityManagementClientSecret"),
               is("u6gMLNLHDv4QRAqqzRezPqcDoohC0vMTvtKUqy5"));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration().get("identityManagementClientId"),
               is("api_platform"));
    assertThat(policy.getPolicyDefinition().getName(), is(policyDefinition.getName()));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(policyDefinition.getTemplateKey()));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), is(policyDefinition.getHttpResourcePointcuts()));
    assertThat(policy.getPolicyDefinition().getApiKeys(), is(policyDefinition.getApiKeys()));
    assertThat(policy.getPolicyDefinition().getOrder(), is(policyDefinition.getOrder()));
    assertThat(policy.getPolicyDefinition().getId(), is(policyDefinition.getId()));
    assertThat(policy.getResolvedTemplate(), is(resolvedTemplate));
  }
}
