/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static com.google.common.collect.Sets.newHashSet;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.CompositePointcut.and;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.CompositePointcut.or;
import static java.util.stream.Collectors.toSet;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes.SourceAttribute.HEADERS;
import static org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes.SourceAttribute.REQUEST_PATH;

import org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes;
import org.mule.runtime.policy.api.PolicyPointcut;

import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;

import java.util.Set;
import java.util.regex.Pattern;

import org.junit.Test;

public class CompositePointcutTestCase {

  private static final PolicyPointcut TRUE = policyPointcutParameters -> true;
  private static final PolicyPointcut FALSE = policyPointcutParameters -> false;

  @Test
  public void orExpression() {
    // A or B

    assertTrue(or(TRUE, TRUE).matches(null));
    assertTrue(or(TRUE, FALSE).matches(null));
    assertTrue(or(FALSE, TRUE).matches(null));
    assertFalse(or(FALSE, FALSE).matches(null));
  }

  @Test
  public void andExpression() {
    // A and B

    assertTrue(and(TRUE, TRUE).matches(null));
    assertFalse(and(TRUE, FALSE).matches(null));
    assertFalse(and(FALSE, TRUE).matches(null));
    assertFalse(and(FALSE, FALSE).matches(null));
  }

  @Test
  public void orAndExpression() {
    // A or (B and C)

    assertTrue(or(TRUE, and(TRUE, TRUE)).matches(null));
    assertTrue(or(TRUE, and(FALSE, TRUE)).matches(null));
    assertTrue(or(TRUE, and(TRUE, FALSE)).matches(null));
    assertTrue(or(TRUE, and(FALSE, FALSE)).matches(null));
    assertTrue(or(FALSE, and(TRUE, TRUE)).matches(null));
    assertFalse(or(FALSE, and(FALSE, TRUE)).matches(null));
    assertFalse(or(FALSE, and(TRUE, FALSE)).matches(null));
    assertFalse(or(FALSE, and(FALSE, FALSE)).matches(null));
  }

  @Test
  public void andOrExpression() {
    // A and (B or C)

    assertTrue(and(TRUE, or(TRUE, TRUE)).matches(null));
    assertTrue(and(TRUE, or(FALSE, TRUE)).matches(null));
    assertTrue(and(TRUE, or(TRUE, FALSE)).matches(null));
    assertFalse(and(TRUE, or(FALSE, FALSE)).matches(null));
    assertFalse(and(FALSE, or(TRUE, TRUE)).matches(null));
    assertFalse(and(FALSE, or(FALSE, TRUE)).matches(null));
    assertFalse(and(FALSE, or(TRUE, FALSE)).matches(null));
    assertFalse(and(FALSE, or(FALSE, FALSE)).matches(null));
  }

  @Test
  public void sourcePolicyAwareAttributesMixedPointcuts() {
    PolicyPointcut pc1 = new HttpHeaderPointcut(POLICY_ID, "h1", "v1");
    PolicyPointcut pc2 = new HttpHeaderPointcut(POLICY_ID, "h2", "v2");
    PolicyPointcut pc3 = new HttpResourcePointcut("/api/.*", "GET");
    PolicyPointcut pc4 = new HttpResourcePointcut("/xapi/.*", "PUT");
    CompositePointcut cp = or(pc1, and(pc2, or(pc3, and(pc4, TRUE))));

    SourcePolicyAwareAttributes attributes = (SourcePolicyAwareAttributes) cp.sourcePolicyAwareAttributes();
    assertThat(attributes.requires(HEADERS), is(true));
    assertThat(attributes.requires(REQUEST_PATH), is(true));
    assertThat(attributes.getHeaders(), is(newHashSet("h1", "h2")));
    assertThat(attributes.getRequestPathPatterns(), hasSize(2));
    Set<String> patternStrings = attributes.getRequestPathPatterns().stream().map(Pattern::toString).collect(toSet());
    assertThat(patternStrings, is(newHashSet("/api/.*", "/xapi/.*")));
  }
}
