/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.listenerPointcutParameters;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.nonHttpPointcutParameters;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.requesterPointcutParameters;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

import com.google.common.collect.ImmutableMap;
import org.mule.runtime.policy.api.PolicyPointcutParameters;

import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.anypoint.tests.logger.TraceLine;
import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;

import org.junit.Before;
import org.junit.Test;

public class HttpResourcePointcutTestCase {

  private MockLogger logger;

  @Before
  public void setUp() {
    logger = new MockLogger();
  }

  @Test
  public void resourceMatchesListenerParameters() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut("/path", "Get");
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = listenerPointcutParameters("", "/path", "Get", ImmutableMap.of());

    boolean matches = pointcut.matches(parameters);

    assertTrue(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request {{}, {}} matches the following resource pointcut: {}", "/path", "Get", pointcut)));
  }

  @Test
  public void resourceMatchesRequestParameters() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut("/path", "Get");
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = requesterPointcutParameters("", "/path", "Get", ImmutableMap.of());

    boolean matches = pointcut.matches(parameters);

    assertTrue(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request {{}, {}} matches the following resource pointcut: {}", "/path", "Get", pointcut)));
  }

  @Test
  public void resourceDifferentCasingMatches() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut("/PATH", "GET");
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = listenerPointcutParameters("", "/path", "Get", ImmutableMap.of());

    boolean matches = pointcut.matches(parameters);

    assertTrue(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request {{}, {}} matches the following resource pointcut: {}", "/path", "Get", pointcut)));
  }

  @Test
  public void regexPatternsMatches() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut(".*", ".*");
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = listenerPointcutParameters("", "/path", "Get", ImmutableMap.of());

    boolean matches = pointcut.matches(parameters);

    assertTrue(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request {{}, {}} matches the following resource pointcut: {}", "/path", "Get", pointcut)));
  }

  @Test
  public void pathMismatches() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut("/path", ".*");
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = listenerPointcutParameters("", "/pathh", "POST", ImmutableMap.of());

    boolean matches = pointcut.matches(parameters);

    assertFalse(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request {{}, {}} does not match the following resource pointcut: {}", "/pathh", "POST",
                                pointcut)));
  }

  @Test
  public void methodMismatches() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut(".*", "GET");
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = listenerPointcutParameters("", "/path", "POST", ImmutableMap.of());

    boolean matches = pointcut.matches(parameters);

    assertFalse(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request {{}, {}} does not match the following resource pointcut: {}", "/path", "POST",
                                pointcut)));
  }

  @Test
  public void invalidParametersType() {
    HttpResourcePointcut pointcut = new HttpResourcePointcut(".*", "GET");
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = nonHttpPointcutParameters("");

    boolean matches = pointcut.matches(parameters);

    assertFalse(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Invalid parameters type found. {}",
                                "org.mule.runtime.policy.api.PolicyPointcutParameters cannot be cast to org.mule.runtime.http.policy.api.HttpPolicyPointcutParameters")));
  }

}
