/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.serialization;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.CONFIG_DATA;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.RESOURCE_POINTCUT;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.RESOURCE_POINTCUTS;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.RESOURCE_POINTCUT_2;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

import org.mule.runtime.core.api.util.IOUtils;
import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.io.File;
import java.io.IOException;

import org.junit.Test;

public class PolicyDefinitionSerializerTestCase extends AbstractMuleTestCase {

  private PolicyDefinitionSerializer policyDefinitionSerializer = new PolicyDefinitionSerializer();

  @Test
  public void serializeFullDefinition() throws IOException {
    File destFile = createTempFile();
    PolicyConfiguration policyConfiguration = new PolicyConfiguration(CONFIG_DATA);
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, RESOURCE_POINTCUTS, 1, policyConfiguration);

    policyDefinitionSerializer.serializeToFile(policyDefinition, destFile);

    assertEquals(IOUtils.getResourceAsString("json/full-definition.json", getClass()).trim(),
                 readFileToString(destFile).trim());
  }

  @Test
  public void serializeWithoutResourcePointcuts() throws IOException {
    File destFile = createTempFile();
    PolicyConfiguration policyConfiguration = new PolicyConfiguration(CONFIG_DATA);
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 1, policyConfiguration);

    policyDefinitionSerializer.serializeToFile(policyDefinition, destFile);

    assertEquals(IOUtils.getResourceAsString("json/no-resource-pointcuts-definition.json", getClass()).trim(),
                 readFileToString(destFile).trim());
  }

  @Test
  public void serializeWithoutConfigData() throws IOException {
    File destFile = createTempFile();
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, RESOURCE_POINTCUTS, 1, new PolicyConfiguration(null));

    policyDefinitionSerializer.serializeToFile(policyDefinition, destFile);

    assertEquals(IOUtils.getResourceAsString("json/no-config-data-definition.json", getClass()).trim(),
                 readFileToString(destFile).trim());
  }

  @Test
  public void deserializeFullDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/full-definition.json").getFile());

    PolicyDefinition policyDefinition = policyDefinitionSerializer.deserializeFromFile(file);

    assertThat(policyDefinition.getName(), is(POLICY_TEMPLATE_KEY.getAssetId() + "-" + POLICY_ID));
    assertThat(policyDefinition.getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertEquals(1, policyDefinition.getApiKeys().size());
    assertEquals(policyDefinition.getApiKeys().get(0), API_KEY);
    assertEquals(policyDefinition.getOrder(), new Integer((1)));
    assertEquals(policyDefinition.getConfigurationData().getConfiguration(), CONFIG_DATA);
    assertEquals(2, policyDefinition.getHttpResourcePointcuts().size());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(0).getMethod(), RESOURCE_POINTCUT.getMethod());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(0).getPath(), RESOURCE_POINTCUT.getPath());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(1).getMethod(), RESOURCE_POINTCUT_2.getMethod());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(1).getPath(), RESOURCE_POINTCUT_2.getPath());
    assertTrue(policyDefinition.isOnline());
  }

  @Test
  public void deserializeFullDefinitionDifferentIdentation() throws IOException {
    File file = new File(getClass().getResource("/json/full-definition-different-identation.json").getFile());

    PolicyDefinition policyDefinition = policyDefinitionSerializer.deserializeFromFile(file);

    assertThat(policyDefinition.getName(), is(POLICY_TEMPLATE_KEY.getAssetId() + "-" + POLICY_ID));
    assertThat(policyDefinition.getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertEquals(1, policyDefinition.getApiKeys().size());
    assertEquals(policyDefinition.getApiKeys().get(0), API_KEY);
    assertEquals(policyDefinition.getOrder(), new Integer((1)));
    assertEquals(policyDefinition.getConfigurationData().getConfiguration(), CONFIG_DATA);
    assertEquals(2, policyDefinition.getHttpResourcePointcuts().size());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(0).getMethod(), RESOURCE_POINTCUT.getMethod());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(0).getPath(), RESOURCE_POINTCUT.getPath());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(1).getMethod(), RESOURCE_POINTCUT_2.getMethod());
    assertEquals(policyDefinition.getHttpResourcePointcuts().get(1).getPath(), RESOURCE_POINTCUT_2.getPath());
    assertTrue(policyDefinition.isOnline());
  }

  @Test
  public void deserializeWithoutResourcePointcuts() throws IOException {
    File file = new File(getClass().getResource("/json/no-resource-pointcuts-definition.json").getFile());

    PolicyDefinition policyDefinition = policyDefinitionSerializer.deserializeFromFile(file);

    assertThat(policyDefinition.getName(), is(POLICY_TEMPLATE_KEY.getAssetId() + "-" + POLICY_ID));
    assertThat(policyDefinition.getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertEquals(1, policyDefinition.getApiKeys().size());
    assertEquals(policyDefinition.getApiKeys().get(0), API_KEY);
    assertEquals(policyDefinition.getOrder(), new Integer((1)));
    assertEquals(policyDefinition.getConfigurationData().getConfiguration(), CONFIG_DATA);
    assertTrue(policyDefinition.getHttpResourcePointcuts().isEmpty());
    assertTrue(policyDefinition.isOnline());
  }

  @Test
  public void deserializeOffline() throws IOException {
    File file = new File(getClass().getResource("/json/offline-single-api-definition.json").getFile());

    OfflinePolicyDefinition policyDefinition = policyDefinitionSerializer.deserializeOfflineFromFile(file);

    assertEquals(policyDefinition.getName(), "offline-single-api-definition");
    assertEquals(1, policyDefinition.getApiKeys().size());
    assertEquals(policyDefinition.getApiKeys().get(0), API_KEY);
    assertEquals(policyDefinition.getOrder(), new Integer((1)));
    assertTrue(policyDefinition.getHttpResourcePointcuts().isEmpty());
    assertFalse(policyDefinition.isOnline());
  }

  @Test
  public void deserializeOfflineMultipleApis() throws IOException {
    File file = new File(getClass().getResource("/json/offline-multiple-apis-definition.json").getFile());

    OfflinePolicyDefinition policyDefinition = policyDefinitionSerializer.deserializeOfflineFromFile(file);

    assertEquals(policyDefinition.getName(), "offline-multiple-apis-definition");
    assertEquals(2, policyDefinition.getApiKeys().size());
    assertEquals(policyDefinition.getApiKeys().get(0), API_KEY);
    assertEquals(policyDefinition.getApiKeys().get(1), API_KEY_2);
    assertEquals(policyDefinition.getOrder(), new Integer((1)));
    assertTrue(policyDefinition.getHttpResourcePointcuts().isEmpty());
    assertFalse(policyDefinition.isOnline());
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeInvalidDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-invalid-field-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeInvalidJson() throws IOException {
    File file = new File(getClass().getResource("/json/offline-invalid-json.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeMissingApiDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-missing-api-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeMissingOrderDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-missing-order-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeInvalidOrder() throws IOException {
    File file = new File(getClass().getResource("/json/offline-invalid-order-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeMissingApiIdDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-missing-api-id-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeMissingTemplateDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-missing-template-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeMissingTemplateGroupIdDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-missing-template-group-id-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeMissingTemplateAssetIdDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-missing-template-asset-id-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  @Test(expected = OfflinePolicyDeserializationException.class)
  public void deserializeMissingTemplateVersionDefinition() throws IOException {
    File file = new File(getClass().getResource("/json/offline-missing-template-version-definition.json").getFile());

    policyDefinitionSerializer.deserializeOfflineFromFile(file);
  }

  private File createTempFile() throws IOException {
    File tempFile = File.createTempFile("pre", "suf");
    tempFile.deleteOnExit();
    return tempFile;
  }
}
