/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus.DEPLOYMENT_SUCCESS;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Collections.emptyMap;
import static java.util.Optional.of;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.deployment.PolicyDeployer;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.notification.PolicyNotificationListenerSuppliers;
import com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore;
import com.mulesoft.mule.runtime.gw.policies.store.EncryptedPropertiesSerializer;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Answers;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class PolicyDeploymentServiceTestCase extends AbstractMuleTestCase {

  private static final String RESOLVED_TEMPLATE = "resolvedTemplate";

  @Rule
  public SystemPropertyTemporaryFolder temporaryFolder = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  @Mock
  private PolicyDeployer policyDeployer;

  @Mock
  private ApiService apiService;

  @Mock(answer = Answers.RETURNS_DEEP_STUBS)
  private Api api;

  private Policy policy;
  private PolicyDefinition policyDefinition;
  private PolicyDeploymentService policyDeploymentService;
  private PolicyDeploymentTracker policyDeploymentTracker;

  @Mock
  private PolicyFactory policyFactory;

  @Mock
  private PolicyTemplate policyTemplate;

  @Before
  public void setUp() {
    when(apiService.get(API_KEY)).thenReturn(of(api));

    PolicyStore policyStore = new DefaultPolicyStore(new EncryptedPropertiesSerializer());

    policyDeploymentTracker = new DefaultPolicyDeploymentTracker();
    policyDeploymentService = new DefaultPolicyDeploymentService(apiService,
                                                                 new PolicyNotificationListenerSuppliers(),
                                                                 policyDeploymentTracker,
                                                                 policyStore);

    overrideVariable("policyDeployer").in(policyDeploymentService).with(policyDeployer);

    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(emptyMap()));
    policy = new Policy(policyTemplate, policyDefinition, RESOLVED_TEMPLATE);

    when(policyFactory.createFromPolicyDefinition(policyDefinition)).thenReturn(policy);
  }

  @Test
  public void newPolicySingleApi() {
    policyDeploymentService.newPolicy(policy);

    verify(policyDeployer).deploy(policy, api);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void newPolicyMultipleApis() {
    ApiImplementation implementation2 = mock(ApiImplementation.class);
    Api api2 = mock(Api.class);
    when(api2.getImplementation()).thenReturn(implementation2);
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, newArrayList(API_KEY, API_KEY_2), null, 1,
                             new PolicyConfiguration(emptyMap()));
    when(apiService.get(API_KEY_2)).thenReturn(of(api2));
    policy = new Policy(policyTemplate, policyDefinition, RESOLVED_TEMPLATE);

    policyDeploymentService.newPolicy(policy);

    verify(policyDeployer).deploy(policy, api);
    verify(policyDeployer).deploy(policy, api2);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void newPolicyForSpecificApi() {
    ApiImplementation implementation2 = mock(ApiImplementation.class);
    Api api2 = mock(Api.class);
    when(api2.getImplementation()).thenReturn(implementation2);
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, newArrayList(API_KEY, API_KEY_2), null, 1,
                             new PolicyConfiguration(emptyMap()));
    when(apiService.get(API_KEY_2)).thenReturn(of(api2));
    policy = new Policy(policyTemplate, policyDefinition, RESOLVED_TEMPLATE);
    policyDeploymentService.newPolicyForApi(policy, API_KEY);

    verify(policyDeployer).deploy(policy, api);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void newPolicyForSpecificNotExistingApi() {
    policyDeploymentService.newPolicyForApi(policy, API_KEY_2);

    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void policyRemoved() {

    policyDeploymentService.removePolicy(policy);

    verify(policyDeployer).undeploy(policyDefinition.getName(), api);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void multipleApisPolicyRemoved() {
    ApiImplementation implementation2 = mock(ApiImplementation.class);
    Api api2 = mock(Api.class);
    when(api2.getImplementation()).thenReturn(implementation2);
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, newArrayList(API_KEY, API_KEY_2), null, 1,
                             new PolicyConfiguration(emptyMap()));
    policy = new Policy(policyTemplate, policyDefinition, RESOLVED_TEMPLATE);
    when(apiService.get(API_KEY_2)).thenReturn(of(api2));

    policyDeploymentService.removePolicy(policy);

    verify(policyDeployer).undeploy(policyDefinition.getName(), api);
    verify(policyDeployer).undeploy(policyDefinition.getName(), api2);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void policyRemovedByName() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy, DEPLOYMENT_SUCCESS));

    policyDeploymentService.removePolicy(policyDefinition.getName());

    verify(policyDeployer).undeploy(policyDefinition.getName(), api);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void policyUpdated() {
    PolicyDefinition updatedDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 2, new PolicyConfiguration(emptyMap())); // Updated order

    Policy updatedPolicy = new Policy(policyTemplate, updatedDefinition, RESOLVED_TEMPLATE);

    policyDeploymentService.updatePolicy(updatedPolicy);

    verify(policyDeployer).undeploy(updatedDefinition.getName(), api);
    verify(policyDeployer).deploy(updatedPolicy, api);
  }
}
