/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.template;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getPolicyTemplatesTempFolder;
import static com.mulesoft.mule.runtime.gw.policies.template.provider.ClasspathPolicyTemplateProvider.POLICY_TEMPLATE_FILE_NAME_PATTERN;
import static java.lang.String.format;
import static org.apache.commons.lang3.reflect.FieldUtils.writeField;
import static org.hamcrest.Matchers.arrayWithSize;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import com.mulesoft.mule.runtime.gw.model.EmptyPolicySpecification;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;
import org.mule.runtime.module.deployment.impl.internal.policy.PolicyTemplateDescriptorFactory;
import org.mule.tck.ZipUtils;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.mule.runtime.gw.policies.template.provider.ClasspathPolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.provider.PolicyTemplateProvider;

import java.io.File;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class ClasspathPolicyTemplateProviderTestCase extends AbstractMuleTestCase {

  @Rule
  public TemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  private PolicyTemplateDescriptor templateDescriptor;

  private PolicyTemplateProvider templateProvider;

  @Before
  public void setUp() throws IllegalAccessException {
    templateProvider = new ClasspathPolicyTemplateProvider();

    PolicyTemplateDescriptorFactory templateDescriptorFactory = mock(PolicyTemplateDescriptorFactory.class);
    templateDescriptor = new PolicyTemplateDescriptor("name");
    templateDescriptor.setRootFolder(getPolicyTemplatesTempFolder());

    when(templateDescriptorFactory.create(any(), any())).thenReturn(templateDescriptor);

    writeField(templateProvider, "templateDescriptorFactory", templateDescriptorFactory, true);
  }

  @Test
  public void provideTemplateFromClasspath() throws Exception {
    File zipWithTemplateInside =
        createZipWithTemplateInside(format(POLICY_TEMPLATE_FILE_NAME_PATTERN, POLICY_TEMPLATE_KEY.getAssetId()));
    addZipToClasspath(zipWithTemplateInside);

    PolicyTemplate policyTemplate = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(policyTemplate.getKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policyTemplate.getTemplateFile(),
               is(new File(getPolicyTemplatesTempFolder(), policyTemplate.getKey().getName() + "/template.xml")));
    assertThat(policyTemplate.getTemplateDescriptor(), is(templateDescriptor));
    assertThat(policyTemplate.getPolicySpecification(), is(instanceOf(EmptyPolicySpecification.class)));
    assertThat(getPolicyTemplatesTempFolder().listFiles(), arrayWithSize(2));
  }

  @Test(expected = IllegalStateException.class)
  public void templateNotPresentInClasspath() {
    templateProvider.provide(POLICY_TEMPLATE_KEY_2);
  }

  private void addZipToClasspath(File zipFile) throws Exception {
    URLClassLoader sysloader = (URLClassLoader) ClassLoader.getSystemClassLoader();
    Method method = URLClassLoader.class.getDeclaredMethod("addURL", URL.class);
    method.setAccessible(true);
    method.invoke(sysloader, new URL("file:" + zipFile.getAbsolutePath()));
  }

  private File createZipWithTemplateInside(String templateFileName) throws Exception {
    File zipFile = new File(muleHome.getRoot(), "zip-with-template-jar-inside.zip");
    File jarFile = new File(muleHome.getRoot(), templateFileName);
    File jarFileContent = new File(muleHome.getRoot(), "template.xml");

    jarFileContent.createNewFile();

    ZipUtils.compress(jarFile, new ZipUtils.ZipResource[] {
        new ZipUtils.ZipResource(jarFileContent.getAbsolutePath(), jarFileContent.getName()),
    });

    ZipUtils.compress(zipFile, new ZipUtils.ZipResource[] {
        new ZipUtils.ZipResource(jarFile.getAbsolutePath(), jarFile.getName())
    });

    return zipFile;
  }
}
