/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.deployment;

import static com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus.DEPLOYMENT_FAILED;
import static com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus.DEPLOYMENT_SUCCESS;

import org.mule.runtime.core.api.policy.PolicyParametrization;

import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyParametrizationFactory;
import com.mulesoft.mule.runtime.gw.policies.notification.PolicyNotificationListenerSuppliers;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentTracker;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;

import org.slf4j.Logger;

public class DefaultPolicyDeployer implements PolicyDeployer {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(DefaultPolicyDeployer.class);

  private final PolicyNotificationListenerSuppliers notificationListenerManager;
  private final PolicyParametrizationFactory policyParametrizationFactory;
  private final PolicyDeploymentTracker policyDeploymentTracker;
  private final PolicyStore policyStore;

  public DefaultPolicyDeployer(PolicyDeploymentTracker policyDeploymentTracker, PolicyStore policyStore,
                               PolicyNotificationListenerSuppliers notificationListenerManager) {
    this.policyParametrizationFactory = new PolicyParametrizationFactory();
    this.notificationListenerManager = notificationListenerManager;
    this.policyDeploymentTracker = policyDeploymentTracker;
    this.policyStore = policyStore;
  }

  @Override
  public void deploy(Policy policy, Api api) {
    PolicyDeploymentStatus status;
    ApiImplementation implementation = api.getImplementation();

    PolicyDefinition policyDefinition = policy.getPolicyDefinition();

    logApplyingPolicy(policyDefinition, api, implementation);
    try {

      policyStore.store(policy);

      PolicyParametrization policyParametrization =
          policyParametrizationFactory.create(policyDefinition, implementation,
                                              policyStore.getPolicyConfigFile(policyDefinition.getName()),
                                              policyStore.getEncryptedPropertiesFile(policyDefinition.getName()),
                                              notificationListenerManager.supply(implementation,
                                                                                 policyDefinition,
                                                                                 policy.getPolicyTemplate()
                                                                                     .getPolicySpecification()));

      implementation.addPolicy(policy.getPolicyTemplate().getTemplateDescriptor(), policyParametrization);

      status = new PolicyDeploymentStatus(policy, DEPLOYMENT_SUCCESS);

      policyStore.cleanDeploymentFailure(policyDefinition, implementation.getApiKey());

      logAppliedPolicy(policyDefinition, api, implementation);
    } catch (Exception exception) {
      status = new PolicyDeploymentStatus(policy, DEPLOYMENT_FAILED);
      DeploymentExceptionHandler deploymentExceptionHandler = new DeploymentExceptionHandler(policyStore);
      deploymentExceptionHandler.handle(policyDefinition, implementation, exception);
    }

    policyDeploymentTracker.policyDeployed(implementation.getApiKey(), status);
  }

  @Override
  public boolean undeploy(String policyName, Api api) {
    ApiImplementation implementation = api.getImplementation();
    String parametrizationId = policyParametrizationFactory.buildParametrizationId(policyName, implementation);

    boolean removed = implementation.removePolicy(parametrizationId);

    if (removed) {
      LOGGER.info("Policy {} un-applied from {} in application {}", policyName, api, implementation.getArtifactName());
    }

    policyDeploymentTracker.policyRemoved(implementation.getApiKey(), policyName);

    if (!policyDeploymentTracker.hasDeployments(policyName)) {
      policyStore.remove(policyName);
    }

    return removed;
  }

  private void logApplyingPolicy(PolicyDefinition policyDefinition, Api api, ApiImplementation implementation) {
    LOGGER.debug("Applying policy {} version {} to {} in application {}", policyDefinition.getName(),
                 policyDefinition.getTemplateKey().getVersion(), api,
                 implementation.getArtifactName());
  }

  private void logAppliedPolicy(PolicyDefinition policyDefinition, Api api, ApiImplementation implementation) {
    LOGGER.info("Applied policy {} version {} to {} in application {}", policyDefinition.getName(),
                policyDefinition.getTemplateKey().getVersion(), api,
                implementation.getArtifactName());
  }
}
