/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class DefaultPolicyDeploymentTracker implements PolicyDeploymentTracker {

  private List<PolicyDeploymentListener> listeners = new LinkedList<>();
  private Map<ApiKey, List<PolicyDeploymentStatus>> policies = new HashMap<>();

  @Override
  public void addPolicyDeploymentListener(PolicyDeploymentListener listener) {
    listeners.add(listener);
  }

  @Override
  public List<Policy> apiRemoved(ApiKey apiKey) {
    return policies.containsKey(apiKey) ? policies.remove(apiKey).stream()
        .map(PolicyDeploymentStatus::getPolicy)
        .collect(toList()) : emptyList();
  }

  @Override
  public void policyRemoved(ApiKey apiKey, String policyName) {
    List<PolicyDeploymentStatus> removedPolicies =
        policies.getOrDefault(apiKey, emptyList()).stream()
            .filter(status -> hasPolicyName(status, policyName))
            .collect(toList());

    policies.getOrDefault(apiKey, emptyList())
        .removeIf(policyStatus -> hasPolicyName(policyStatus, policyName));

    removedPolicies
        .forEach(status -> listeners.forEach(listener -> listener.policyRemoved(apiKey, status)));
  }

  @Override
  public void policyDeployed(ApiKey apiKey, PolicyDeploymentStatus policyDeploymentStatus) {
    policies.putIfAbsent(apiKey, new ArrayList<>());
    policies.get(apiKey).add(policyDeploymentStatus);
    listeners.forEach(listener -> listener.policyDeployed(apiKey, policyDeploymentStatus));
  }

  @Override
  public List<PolicyDeploymentStatus> onlinePolicyStatuses(ApiKey apiKey) {
    return policyStatuses(apiKey).stream()
        .filter(status -> status.getPolicy().getPolicyDefinition().isOnline())
        .collect(toList());
  }

  @Override
  public List<PolicyDeploymentStatus> policyStatuses(ApiKey apiKey) {
    return policies.getOrDefault(apiKey, emptyList());
  }

  @Override
  public Optional<Policy> findByName(String policyName) {
    return policies.values().stream()
        .flatMap(Collection::stream)
        .filter(policyStatus -> policyStatus.getPolicy().getPolicyDefinition().getName().equals(policyName))
        .map(PolicyDeploymentStatus::getPolicy)
        .findFirst();
  }

  @Override
  public boolean hasDeployments(String policyName) {
    return policies.values().stream()
        .flatMap(Collection::stream)
        .anyMatch(policyStatus -> policyStatus.getPolicy().getPolicyDefinition().getName().equals(policyName));
  }

  private boolean hasPolicyName(PolicyDeploymentStatus status, String name) {
    return status.getPolicy().getPolicyDefinition().getName().equals(name);
  }

}
