/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.store;

import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.createDirectoryIfNecessary;
import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getPoliciesFolder;
import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.policies.store.EncryptedPropertiesSerializer.ENCRYPTED_PROPERTIES_YAML_FILE_NAME;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.deleteDirectory;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.lang3.exception.ExceptionUtils.getStackTrace;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.serialization.OfflinePolicyDeserializationException;
import com.mulesoft.mule.runtime.gw.policies.serialization.PolicyDefinitionSerializer;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

/**
 * Default implementation that stores the policies in the file system
 */
public class DefaultPolicyStore implements PolicyStore {

  public static final String POLICY_DEFINITION_JSON_FILE_NAME = "policy-definition.json";
  public static final String POLICY_CONFIG_XML_FILE_NAME = "policy.xml";

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(DefaultPolicyStore.class);

  private final PolicyStoreFiles storeFiles;

  private PolicyDefinitionSerializer policyDefinitionSerializer;
  private EncryptedPropertiesSerializer encryptedPropertiesSerializer;

  public DefaultPolicyStore(EncryptedPropertiesSerializer encryptedPropertiesSerializer) {
    this.policyDefinitionSerializer = new PolicyDefinitionSerializer();
    this.storeFiles = new PolicyStoreFiles(getPoliciesFolder());
    this.encryptedPropertiesSerializer = encryptedPropertiesSerializer;
  }

  @Override
  public List<PolicyDefinition> load() {
    return loadPolicyDefinitions(storeFiles.listPolicyFolders());
  }

  @Override
  public List<PolicyDefinition> onlinePolicies() {
    return load().stream().filter(PolicyDefinition::isOnline).collect(toList());
  }

  @Override
  public List<PolicyDefinition> offlinePolicies() {
    return storeFiles.listOfflinePoliciesDescriptors().stream()
        .map(file -> {
          try {
            return policyDefinitionSerializer.deserializeOfflineFromFile(file);
          } catch (OfflinePolicyDeserializationException e) {
            return null;
          }
        })
        .filter(Objects::nonNull)
        .collect(toList());
  }

  @Override
  public boolean contains(String policyName) {
    return storeFiles.getPolicyFolder(policyName).exists();
  }

  @Override
  public File getPolicyConfigFile(String policyName) {
    return new File(storeFiles.getPolicyFolder(policyName), POLICY_CONFIG_XML_FILE_NAME);
  }

  @Override
  public File getEncryptedPropertiesFile(String policyName) {
    return new File(storeFiles.getPolicyFolder(policyName), ENCRYPTED_PROPERTIES_YAML_FILE_NAME);
  }

  @Override
  public boolean remove(String policyName) {
    try {
      if (contains(policyName)) {
        deleteDirectory(storeFiles.getPolicyFolder(policyName));
      }

      cleanDeploymentFailures(policyName);

      return true;
    } catch (IOException e) {
      LOGGER.warn("Error trying to delete policy folder {}. {}", policyName, errorMessage(e));
    }

    return false;
  }

  @Override
  public void store(PolicyDefinition policyDefinition) {
    File destFolder = storeFiles.getPolicyFolder(policyDefinition.getName());
    createDirectoryIfNecessary(destFolder);

    try {
      storePolicyDefinition(policyDefinition, destFolder);
    } catch (IOException e) {
      throw new IllegalStateException("Unexpected error storing policy " + policyDefinition.getName(), e);
    }
  }

  @Override
  public void store(Policy policy) {
    File destFolder = storeFiles.getPolicyFolder(policy.getPolicyDefinition().getName());
    createDirectoryIfNecessary(destFolder);

    try {
      storePolicyDefinition(policy.getPolicyDefinition(), destFolder);
      storePolicyConfiguration(policy.getResolvedTemplate(), destFolder);
      encryptedPropertiesSerializer.store(policy.getConfigFile(), destFolder);
    } catch (IOException e) {
      throw new IllegalStateException("Unexpected error storing policy " + policy.getPolicyDefinition().getName(), e);
    }
  }

  @Override
  public void storeDeploymentFailure(PolicyDefinition policyDefinition, ApiKey apiKey, Exception exception) {
    String fileName = deploymentFailureFileName(policyDefinition, apiKey);
    File failureFile = new File(storeFiles.getFailedPoliciesFolder(), fileName);

    try {
      FileUtils.write(failureFile, "Error deploying policy " + policyDefinition.getName() + " to " + apiKey + ":\n");
      FileUtils.write(failureFile, getStackTrace(exception), true);
    } catch (IOException e) {
      LOGGER.warn("Could not write failure to failed-policies for {}. {}", policyDefinition.getName(), errorMessage(e));
    }
  }

  @Override
  public void cleanDeploymentFailure(PolicyDefinition policyDefinition, ApiKey apiKey) {
    String fileName = deploymentFailureFileName(policyDefinition, apiKey);
    deleteQuietly(new File(storeFiles.getFailedPoliciesFolder(), fileName));

    if (storeFiles.listPolicyDeploymentFailures().isEmpty()) {
      deleteQuietly(storeFiles.getFailedPoliciesFolder());
    }
  }

  @Override
  public void cleanDeploymentFailures(String policyName) {
    storeFiles.listPolicyDeploymentFailures().stream()
        .filter(file -> policyName.equals(StringUtils.substringBefore(file.getName(), "@")))
        .forEach(FileUtils::deleteQuietly);

    if (storeFiles.listPolicyDeploymentFailures().isEmpty()) {
      deleteQuietly(storeFiles.getFailedPoliciesFolder());
    }
  }

  private List<PolicyDefinition> loadPolicyDefinitions(List<File> policyFolders) {
    return policyFolders.stream()
        .map(this::loadPolicyDefinition)
        .flatMap(o -> o.map(Stream::of).orElseGet(Stream::empty))
        .collect(toList());
  }

  private Optional<PolicyDefinition> loadPolicyDefinition(File policyFolder) {
    try {
      File policyFile = new File(policyFolder, POLICY_DEFINITION_JSON_FILE_NAME);
      return of(policyDefinitionSerializer.deserializeFromFile(policyFile));
    } catch (Exception e) {
      LOGGER.error("Could not read policy from JSON file in {}. {}", policyFolder.getName(), errorMessage(e));
    }

    return empty();
  }

  private void storePolicyConfiguration(String resolvedTemplate, File destFolder) throws IOException {
    if (resolvedTemplate != null) {
      FileUtils.write(new File(destFolder, POLICY_CONFIG_XML_FILE_NAME), resolvedTemplate);
    }
  }

  private void storePolicyDefinition(PolicyDefinition policyDefinition, File destFolder) throws IOException {
    policyDefinitionSerializer.serializeToFile(policyDefinition, new File(destFolder, POLICY_DEFINITION_JSON_FILE_NAME));
  }

  private String deploymentFailureFileName(PolicyDefinition policyDefinition, ApiKey apiKey) {
    return policyDefinition.getName() + "@" + apiKey.id() + ".log";
  }

}
