/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.offline;

import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getOfflinePoliciesFolder;
import static com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore.POLICY_CONFIG_XML_FILE_NAME;
import static com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore.POLICY_DEFINITION_JSON_FILE_NAME;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static junit.framework.TestCase.assertTrue;
import static org.apache.commons.io.FileUtils.copyFile;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.runtime.api.exception.MuleException;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;
import org.mule.tck.probe.PollingProber;

import com.mulesoft.anypoint.tests.DescriptiveProbe;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.policies.factory.DefaultPolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.template.provider.FileSystemPolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.resolver.HandlebarsPolicyTemplateResolver;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;
import com.mulesoft.mule.runtime.gw.reflection.VariableOverride;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class OfflinePolicyWatcherTestCase extends AbstractMuleTestCase {

  @Rule
  public SystemPropertyTemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private PolicyDeploymentService policyDeploymentService;
  private MockLogger logger;

  @Before
  public void setUp() throws MuleException {
    policyDeploymentService = mock(PolicyDeploymentService.class);

    OfflinePolicyWatcher offlinePolicyWatcher = new OfflinePolicyWatcher(policyDeploymentService, mock(PolicyFactory.class));
    overrideLogger(offlinePolicyWatcher);

    offlinePolicyWatcher.initialise();
    offlinePolicyWatcher.start();
  }

  @Test
  public void newFileIsDetected() throws URISyntaxException, IOException {
    File offlineDefinitionFile = new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME);

    copyFile(definitionFile(), offlineDefinitionFile);

    new PollingProber(2000, 200).check(new DescriptiveProbe(() -> {
      verify(policyDeploymentService).newPolicy(any());
      verifyNoMoreInteractions(policyDeploymentService);
    }));
  }

  @Test
  public void invalidDefinitionNotDeployed() throws URISyntaxException, IOException, InterruptedException {
    File offlineDefinitionFile = new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME);

    copyFile(invalidDefinitionFile(), offlineDefinitionFile);

    new PollingProber(2000, 200).check(new DescriptiveProbe(() -> {
      verifyNoMoreInteractions(policyDeploymentService);
    }));
  }

  @Test
  public void invalidExtensionInNewFileIsIgnored() throws URISyntaxException, IOException, InterruptedException {
    File invalidFile = new File(getOfflinePoliciesFolder(), POLICY_CONFIG_XML_FILE_NAME);
    copyFile(invalidExtensionFile(), invalidFile);

    new PollingProber(2000, 200).check(new DescriptiveProbe(() -> {
      assertTrue(logger.lines().stream()
          .anyMatch(logLine -> logLine.toString().contains("Invalid extension")));
      verifyNoMoreInteractions(policyDeploymentService);
    }));
  }

  @Test
  public void modificationIsDetected() throws URISyntaxException, IOException {
    copyFile(definitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    new PollingProber(2000, 200).check(new DescriptiveProbe(() -> verify(policyDeploymentService).newPolicy(any())));

    copyFile(updatedDefinitionFile(), new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME));

    new PollingProber(2000, 200).check(new DescriptiveProbe(() -> {
      verify(policyDeploymentService).newPolicy(any());
      verify(policyDeploymentService).updatePolicy(any());
      verifyNoMoreInteractions(policyDeploymentService);
    }));
  }

  @Test
  public void removeIsDetected() throws URISyntaxException, IOException {
    File offlineDefinitionFile = new File(getOfflinePoliciesFolder(), POLICY_DEFINITION_JSON_FILE_NAME);
    copyFile(definitionFile(), offlineDefinitionFile);

    new PollingProber(2000, 200).check(new DescriptiveProbe(() -> verify(policyDeploymentService).newPolicy(any())));

    boolean deleted = offlineDefinitionFile.delete();

    assertTrue(deleted);
    new PollingProber(2000, 200).check(new DescriptiveProbe(() -> {
      verify(policyDeploymentService).newPolicy(any());
      verify(policyDeploymentService).removePolicy("policy-definition");
      verifyNoMoreInteractions(policyDeploymentService);
    }));
  }

  private File definitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/full-definition.json").toURI());
  }

  private File updatedDefinitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/updated-definition.json").toURI());
  }

  private File invalidDefinitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/offline-invalid-field-definition.json").toURI());
  }

  private File invalidExtensionFile() throws URISyntaxException {
    return new File(getClass().getResource("/templates/template.xml").toURI());
  }

  private void overrideLogger(OfflinePolicyWatcher offlinePolicyWatcher) {
    logger = new MockLogger();
    OfflinePolicyListener offlinePolicyListener = new Inspector(offlinePolicyWatcher).read("listener");
    VariableOverride.overrideLogger().in(offlinePolicyListener).with(logger);
  }

}
