/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static java.util.Collections.singleton;
import static java.util.Optional.empty;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.when;
import static org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes.SourceAttribute.HEADERS;
import static org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes.SourceAttribute.REQUEST_PATH;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;
import org.mule.runtime.http.policy.api.HttpPolicyPointcutParameters;
import org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class HttpHeaderPointcutTestCase {

  private static final String POLICY_NAME = "name";

  @Mock
  private HttpPolicyPointcutParameters parameters;

  private MultiMap<String, String> requestHeaders = new CaseInsensitiveMultiMap();

  @Before
  public void setUp() {
    when(parameters.getSourceParameters()).thenReturn(empty());
    when(parameters.getHeaders()).thenReturn(requestHeaders);
  }

  @Test
  public void headerNotPresent() {
    HttpHeaderPointcut pointcut = new HttpHeaderPointcut(POLICY_NAME, "h1", ".*");
    assertThat(pointcut.matches(parameters), is(false));
  }

  @Test
  public void matchOnlyValue() {
    HttpHeaderPointcut pointcut = new HttpHeaderPointcut(POLICY_NAME, "h1", "v1");
    requestHeaders.put("h1", "v1");
    assertThat(pointcut.matches(parameters), is(true));
  }

  @Test
  public void matchOneOfMultipleValues() {
    HttpHeaderPointcut pointcut = new HttpHeaderPointcut(POLICY_NAME, "h1", ".2");
    requestHeaders.put("h1", "v1");
    requestHeaders.put("h1", "v2");
    requestHeaders.put("h1", "v3");
    assertThat(pointcut.matches(parameters), is(true));
  }

  @Test
  public void differentCaseNameMatches() {
    HttpHeaderPointcut pointcut = new HttpHeaderPointcut(POLICY_NAME, "Hd1", "v1");
    requestHeaders.put("hD1", "v1");
    assertThat(pointcut.matches(parameters), is(true));
  }

  @Test
  public void differentCaseValueDoesNotMatch() {
    HttpHeaderPointcut pointcut = new HttpHeaderPointcut(POLICY_NAME, "h1", "V1");
    requestHeaders.put("h1", "v1");
    assertThat(pointcut.matches(parameters), is(false));
  }

  @Test
  public void substringDoesNotMatch() {
    HttpHeaderPointcut pointcut = new HttpHeaderPointcut(POLICY_NAME, "h1", "v1");
    requestHeaders.put("h1", "v11");
    assertThat(pointcut.matches(parameters), is(false));
  }

  @Test
  public void sourcePolicyAwareAttributes() {
    HttpHeaderPointcut pointcut = new HttpHeaderPointcut(POLICY_NAME, "h1", "v1");
    SourcePolicyAwareAttributes policyAwareAttributes = (SourcePolicyAwareAttributes) pointcut.sourcePolicyAwareAttributes();
    assertThat(policyAwareAttributes.requires(HEADERS), is(true));
    assertThat(policyAwareAttributes.requires(REQUEST_PATH), is(false));
    assertThat(policyAwareAttributes.getHeaders(), is(singleton("h1")));
  }

}
