/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getOfflinePoliciesFolder;
import static com.mulesoft.mule.runtime.gw.model.PolicySet.PolicySetOrigin.PLATFORM;
import static com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus.DEPLOYMENT_SUCCESS;
import static com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus.TEMPLATE_DOWNLOAD_FAILED;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static com.mulesoft.anypoint.retry.BackoffRunnableRetrier.zeroDelayOnScheduling;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Optional.of;
import static java.util.stream.IntStream.range;
import static org.apache.commons.io.FileUtils.copyFileToDirectory;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.runtime.deployment.model.api.application.ApplicationPolicyManager;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.ErrorLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.anypoint.tests.scheduler.ObservableScheduledExecutorService;
import com.mulesoft.anypoint.tests.scheduler.observer.RunnableLoggerObserver;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.anypoint.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.anypoint.backoff.scheduler.factory.FixedExecutorBackoffSchedulerFactory;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus.DeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.deployment.DeploymentExceptionHandler;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.PolicySetDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore;
import com.mulesoft.mule.runtime.gw.policies.store.EncryptedPropertiesSerializer;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;
import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateAssetException;
import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateResolverException;
import com.mulesoft.anypoint.retry.BackoffRunnableRetrier;
import com.mulesoft.anypoint.retry.RunnableRetrier;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Answers;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class PolicySetDeploymentServiceTestCase extends AbstractMuleTestCase {

  private static final String RESOLVED_TEMPLATE = "resolvedTemplate";

  @Rule
  public SystemPropertyTemporaryFolder temporaryFolder = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  @Mock
  private PolicySetDeploymentListener listener;

  @Mock
  private ApplicationPolicyManager policyManager;

  @Mock
  private PolicyDeploymentService policyDeploymentService;
  private PolicyStore policyStore;
  private PolicyDeploymentTracker policyDeploymentTracker;

  @Mock
  private PolicyFactory policyFactory;

  private Policy policy1;
  private Policy policy2;
  private PolicyDefinition policyDefinition1;
  private PolicyDefinition policyDefinition2;
  private RunnableRetrier<ApiKey> runnableRetrier;

  private RunnableLoggerObserver executorLogger;
  private PolicySetDeploymentService policySetDeploymentService;
  private MockLogger logger;

  @Mock
  private ApiService apiService;

  @Mock(answer = Answers.RETURNS_DEEP_STUBS)
  private Api api;

  @Before
  public void setUp() {
    logger = new MockLogger();

    ApiImplementation apiImplementation = mock(ApiImplementation.class);
    when(api.getImplementation()).thenReturn(apiImplementation);
    when(apiImplementation.getArtifactName()).thenReturn("test-app");
    when(apiImplementation.getApiKey()).thenReturn(API_KEY);

    apiService = mock(ApiService.class);
    when(apiService.get(API_KEY)).thenReturn(of(api));

    policyDeploymentTracker = new DefaultPolicyDeploymentTracker();

    executorLogger = new RunnableLoggerObserver();
    runnableRetrier = runnableRetrier(new GatewayConfiguration());
    policyStore = new DefaultPolicyStore(new EncryptedPropertiesSerializer());
    policySetDeploymentService = policySetDeploymentService(policyDeploymentTracker);

    PolicyConfiguration policyConfiguration = new PolicyConfiguration(emptyMap());
    PolicyConfiguration policyConfiguration2 = new PolicyConfiguration(emptyMap());

    policyDefinition1 = new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, policyConfiguration);
    policyDefinition2 = new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY, API_KEY, null, 1, policyConfiguration2);

    PolicyTemplate template = mock(PolicyTemplate.class);
    policy1 = new Policy(template, policyDefinition1, RESOLVED_TEMPLATE);
    policy2 = new Policy(template, policyDefinition2, RESOLVED_TEMPLATE);


    when(policyFactory.createFromPolicyDefinition(policyDefinition1)).thenReturn(policy1);
    when(policyFactory.createFromPolicyDefinition(policyDefinition2)).thenReturn(policy2);

    overrideLogger().in(policySetDeploymentService).with(logger);
    overrideLogger().in(new DeploymentExceptionHandler(null)).with(logger);
  }

  private DefaultPolicySetDeploymentService policySetDeploymentService(PolicyDeploymentTracker policyDeploymentTracker) {
    return new DefaultPolicySetDeploymentService(runnableRetrier, policyDeploymentService, policyDeploymentTracker, policyStore,
                                                 policyFactory, apiService);
  }

  @Test
  public void newPoliciesInPolicySet() {

    policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition1, policyDefinition2), null));

    verify(policyDeploymentService).newPolicy(policy1);
    verify(policyDeploymentService).newPolicy(policy2);
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void onePolicyRemovedInPolicySet() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy2, DEPLOYMENT_SUCCESS));

    policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition1), null));

    verify(policyDeploymentService).removePolicy(policy2.getPolicyDefinition().getName());
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void everyPolicyRemovedInPolicySet() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy2, DEPLOYMENT_SUCCESS));

    policiesForApi(API_KEY, new PolicySet(newArrayList(), null));

    verify(policyDeploymentService).removePolicy(policy1.getPolicyDefinition().getName());
    verify(policyDeploymentService).removePolicy(policy2.getPolicyDefinition().getName());
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void onePolicyUpdatedInPolicySet() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy2, DEPLOYMENT_SUCCESS));
    PolicyDefinition updatedPolicyDefinition =
        new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY, API_KEY, null, 10, new PolicyConfiguration(emptyMap()));
    when(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition))
        .thenReturn(new Policy(mock(PolicyTemplate.class), updatedPolicyDefinition, RESOLVED_TEMPLATE));

    policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition1, updatedPolicyDefinition), null));

    verify(policyDeploymentService).updatePolicy(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition));
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void everyPolicyUpdatedInPolicySet() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy2, DEPLOYMENT_SUCCESS));
    PolicyDefinition updatedPolicyDefinition1 =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 10, new PolicyConfiguration(emptyMap()));
    PolicyDefinition updatedPolicyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY, API_KEY, null, 11, new PolicyConfiguration(emptyMap()));

    PolicyTemplate template = mock(PolicyTemplate.class);
    when(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition1))
        .thenReturn(new Policy(template, updatedPolicyDefinition1, RESOLVED_TEMPLATE));
    when(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition2))
        .thenReturn(new Policy(template, updatedPolicyDefinition2, RESOLVED_TEMPLATE));

    policiesForApi(API_KEY, new PolicySet(newArrayList(updatedPolicyDefinition1, updatedPolicyDefinition2), null));


    verify(policyDeploymentService).updatePolicy(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition1));
    verify(policyDeploymentService).updatePolicy(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition2));
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void onePolicyUpdatedAndOneDeletedInPolicySet() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy2, DEPLOYMENT_SUCCESS));
    PolicyDefinition updatedPolicyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 10, new PolicyConfiguration(emptyMap()));
    when(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition))
        .thenReturn(new Policy(mock(PolicyTemplate.class), updatedPolicyDefinition, RESOLVED_TEMPLATE));

    policiesForApi(API_KEY, new PolicySet(newArrayList(updatedPolicyDefinition), null));

    verify(policyDeploymentService).updatePolicy(policyFactory.createFromPolicyDefinition(updatedPolicyDefinition));
    verify(policyDeploymentService).removePolicy(policy2.getPolicyDefinition().getName());
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void removeApi() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy2, DEPLOYMENT_SUCCESS));

    policySetDeploymentService.removeAll(API_KEY);

    verify(policyDeploymentService).removePolicy(policy1);
    verify(policyDeploymentService).removePolicy(policy2);
    verifyNoMoreInteractions(policyDeploymentService);
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(0));
  }

  @Test
  public void removeApiAndReAddIt() {
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy2, DEPLOYMENT_SUCCESS));

    policySetDeploymentService.removeAll(API_KEY);
    policiesForApi(API_KEY, new PolicySet(newArrayList(policyDefinition1), null));

    verify(policyDeploymentService).newPolicy(policy1);
    verify(policyDeploymentService).removePolicy(policy1);
    verify(policyDeploymentService).removePolicy(policy2);
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void deploymentListenerWhenEveryPolicyIsOk() {
    policySetDeploymentService.addPolicyDeploymentListener(listener);
    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1), null);

    policiesForApi(API_KEY, policySet);

    verify(listener).onPolicySetDeploymentCompleted(eq(API_KEY), eq(policySet), any());

    policySetDeploymentService.removeAll(API_KEY);

    verify(listener).onPoliciesRemoved(API_KEY);
  }

  @Test
  public void onRedeploymentStartListenersAreNotNotified() {
    ApiImplementation implementation = mock(ApiImplementation.class);
    when(implementation.getApiKey()).thenReturn(API_KEY);
    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1), null);
    policySetDeploymentService.addPolicyDeploymentListener(listener);
    policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy1, DEPLOYMENT_SUCCESS));

    policiesForApi(API_KEY, policySet);
    policySetDeploymentService.onApiRedeploymentStart(implementation);

    verify(listener).onPoliciesRemoved(API_KEY);
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(0));
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void deploymentListenerWhenSomeErrorInPolicy() {
    PolicySet policySet = policiesWithStatus(TEMPLATE_DOWNLOAD_FAILED, DEPLOYMENT_SUCCESS);

    policiesForApi(API_KEY, policySet);

    verify(policyDeploymentService).newPolicy(policy1);
    verify(policyDeploymentService).newPolicy(policy2);
    verify(policyDeploymentService, never()).removePolicy(policy1);
    verify(policyDeploymentService, never()).removePolicy(policy2);
    verifyZeroInteractions(listener);
  }

  @Test
  public void policiesForApiScheduleTask() {
    policySetDeploymentService.policiesForApi(API_KEY, policiesWithStatus(DEPLOYMENT_SUCCESS, DEPLOYMENT_SUCCESS));

    verifyZeroInteractions(policyDeploymentService);
    assertThat(executorLogger.scheduledTasks(), hasSize(1));
  }

  @Test
  public void retryWhenDownloadErrorPolicy() {
    PolicySet policySet = policiesWithStatus(TEMPLATE_DOWNLOAD_FAILED, DEPLOYMENT_SUCCESS);
    policiesForApi(API_KEY, policySet);
    onDeploySetDeploymentStatus(policyDefinition1, policy1, DEPLOYMENT_SUCCESS);

    retryTask(1).run();

    assertThat(executorLogger.scheduledTasks(), hasSize(2));
    verify(policyDeploymentService).newPolicy(policy2);
    verify(policyDeploymentService).newPolicy(policy1);
    verify(policyDeploymentService).updatePolicy(policy1);
    verify(listener).onPolicySetDeploymentCompleted(eq(API_KEY), eq(policySet), any());
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void retryWhenFatalErrorOnDeployment() {
    PolicyDeploymentTracker mockTracker = mock(PolicyDeploymentTracker.class);
    doThrow(new InternalError()).doReturn(new ArrayList<PolicyDeploymentStatus>()).when(mockTracker)
        .onlinePolicyStatuses(API_KEY);
    overrideVariable("policyDeploymentTracker").in(policySetDeploymentService).with(mockTracker);

    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1), null);
    policiesForApi(API_KEY, policySet);

    verify(mockTracker).onlinePolicyStatuses(API_KEY);
    verifyZeroInteractions(policyDeploymentService);

    policySetDeploymentService.policiesForApi(API_KEY, policySet);
    retryTask(1).run();

    verify(mockTracker, times(3)).onlinePolicyStatuses(API_KEY);
    verify(policyDeploymentService).newPolicy(policy1);
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void multipleRetriesAreRequiredForDownloadToBeSuccessful() {
    PolicySet policySet = policiesWithStatus(TEMPLATE_DOWNLOAD_FAILED, DEPLOYMENT_SUCCESS);
    policiesForApi(API_KEY, policySet);
    range(1, 11).forEach(i -> retryTask(i).run());
    onDeploySetDeploymentStatus(policyDefinition1, policy1, DEPLOYMENT_SUCCESS);

    retryTask(11).run();

    assertThat(executorLogger.scheduledTasks(), hasSize(12));
    verify(policyDeploymentService, never()).removePolicy(policy2);
    verify(policyDeploymentService).newPolicy(policy2);
    verify(policyDeploymentService, times(11)).updatePolicy(policy1);
    verify(policyDeploymentService).newPolicy(policy1);
    verify(listener).onPolicySetDeploymentCompleted(eq(API_KEY), eq(policySet), any());
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void differentPoliciesRecoversAtDifferentSpeeds() {
    /*
     * Policy 1 will fail to deploy 10 times, Policy 2 20 times. Policy deployment listener should be called only once after the
     * second policy's template is downloaded successfully.
     */
    PolicySet policySet = policiesWithStatus(TEMPLATE_DOWNLOAD_FAILED, TEMPLATE_DOWNLOAD_FAILED);
    policiesForApi(API_KEY, policySet);

    range(1, 11).forEach(i -> retryTask(i).run());
    onDeploySetDeploymentStatus(policyDefinition1, policy1, DEPLOYMENT_SUCCESS);

    range(11, 21).forEach(i -> retryTask(i).run());
    verifyZeroInteractions(listener);

    onDeploySetDeploymentStatus(policyDefinition2, policy2, DEPLOYMENT_SUCCESS);
    retryTask(21).run();

    assertThat(executorLogger.scheduledTasks(), hasSize(22));
    verify(policyDeploymentService).newPolicy(policy1);
    verify(policyDeploymentService).newPolicy(policy2);
    verify(policyDeploymentService, times(11)).updatePolicy(policy1);
    verify(policyDeploymentService, times(21)).updatePolicy(policy2);
    verify(listener).onPolicySetDeploymentCompleted(eq(API_KEY), eq(policySet), any());
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void logWhenSomeTemplateFailsToBeDownloaded() {
    policiesForApi(API_KEY, policiesWithStatus(TEMPLATE_DOWNLOAD_FAILED, TEMPLATE_DOWNLOAD_FAILED));
    List<String> policiesNames = new ArrayList<>();
    policiesNames.add(policyDefinition1.getName());
    policiesNames.add(policyDefinition2.getName());

    assertThat(logger.lines(), hasSize(4));
    assertThat(logger.lines().get(0),
               is(new DebugLine("Deploying policies {} from {} to API {}", policiesNames, "Platform", API_KEY)));
    assertThat(logger.lines().get(1), is(new DebugLine("New policy {} detected to apply", policyDefinition1.getName())));
    assertThat(logger.lines().get(2), is(new DebugLine("New policy {} detected to apply", policyDefinition2.getName())));
    assertThat(logger.lines().get(3), is(new DebugLine("Template download failed for API {} - Policies {}.", API_KEY,
                                                       policiesNames)));
  }

  @Test
  public void exceptionOnListenerDoesNotBlockOtherListeners() {
    PolicySetDeploymentListener listener2 = mock(PolicySetDeploymentListener.class);
    doThrow(new RuntimeException()).when(listener).onPolicySetDeploymentCompleted(any(), any(), any());
    doThrow(new RuntimeException()).when(listener).onPoliciesRemoved(any());
    policySetDeploymentService.addPolicyDeploymentListener(listener);
    policySetDeploymentService.addPolicyDeploymentListener(listener2);
    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1), null);

    policiesForApi(API_KEY, policySet);

    verify(listener).onPolicySetDeploymentCompleted(eq(API_KEY), eq(policySet), any());
    verify(listener2).onPolicySetDeploymentCompleted(eq(API_KEY), eq(policySet), any());

    policySetDeploymentService.removeAll(API_KEY);

    verify(listener).onPoliciesRemoved(API_KEY);
    verify(listener2).onPoliciesRemoved(API_KEY);
  }

  @Test
  public void conciliatePolicies() {
    policyStore.store(policyDefinition1);

    policySetDeploymentService.conciliatePolicies(API_KEY, newArrayList(policyDefinition2));

    assertThat(policyStore.load(), hasSize(1));
    assertThat(policyStore.load().get(0), is(policyDefinition2));
  }

  @Test
  public void conciliateUpdatedPolicy() {
    policyStore.store(policyDefinition1);
    int differentOrder = 15;
    PolicyDefinition policyDefinition1Updated =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, differentOrder, new PolicyConfiguration(emptyMap()));

    policySetDeploymentService.conciliatePolicies(API_KEY, newArrayList(policyDefinition1Updated));

    assertThat(policyStore.load(), hasSize(1));
    assertThat(policyStore.load().get(0), is(policyDefinition1Updated));
  }

  @Test
  public void initApiDeploysOfflineStoredPolicies() throws URISyntaxException, IOException {
    OfflinePolicyDefinition offlinePolicyDefinition =
        new OfflinePolicyDefinition("offline-single-api-definition", POLICY_TEMPLATE_KEY, API_KEY, null, 1,
                                    new PolicyConfiguration(emptyMap()));
    policyStore.store(policyDefinition1);
    copyFileToDirectory(definitionFile(), getOfflinePoliciesFolder());
    Api api = mock(Api.class);
    when(api.getKey()).thenReturn(API_KEY);

    policySetDeploymentService.onApiDeploymentSuccess(api);

    verify(policyDeploymentService).newPolicyForApi(policyFactory.createFromPolicyDefinition(offlinePolicyDefinition), API_KEY);
    verifyNoMoreInteractions(policyDeploymentService);
  }

  @Test
  public void deployPolicyFailsWhenDownloadingTemplate() {
    PolicyTemplateAssetException exception = new PolicyTemplateAssetException("", new RuntimeException());
    when(policyFactory.createFromPolicyDefinition(policyDefinition1)).thenThrow(exception);

    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1), null);
    policiesForApi(API_KEY, policySet);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).isDeploymentSuccess(), is(false));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).isTemplateDownloadFailed(), is(true));
  }

  @Test
  public void deployPolicyWithMissingTemplateStillStoresPolicy() throws PolicyTemplateResolverException {
    PolicyTemplateResolverException exception = new PolicyTemplateResolverException("", new RuntimeException());
    when(policyFactory.createFromPolicyDefinition(policyDefinition1)).thenThrow(exception);

    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1), null);
    policiesForApi(API_KEY, policySet);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).isDeploymentSuccess(), is(false));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).isTemplateDownloadFailed(), is(false));
    verifyZeroInteractions(policyManager);
  }

  @Test
  public void deployPolicyThrowingIllegalStateExceptionLogsException() throws PolicyTemplateResolverException {
    IllegalStateException exception =
        new IllegalStateException("More than one directory under ~/runtimePath/.mule/policy-templates/wrongPolicy/META-INF/maven so pom.xml file for artifact in folder ~/runtimePath/.mule/policy-templates/wrongPolicy could not be found");
    when(policyFactory.createFromPolicyDefinition(policyDefinition1)).thenThrow(exception);

    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1), PLATFORM);
    policiesForApi(API_KEY, policySet);

    List<String> policiesNames = asList(policyDefinition1.getName());

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).isDeploymentFailed(), is(true));
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0),
               is(new DebugLine("Deploying policies {} from {} to API {}", policiesNames, "Platform", API_KEY)));
    assertThat(logger.lines().get(1),
               is(new ErrorLine("Error deploying policy " + policiesNames.get(0) + " to application " + "test-app", exception)));
    verifyZeroInteractions(policyDeploymentService);
  }

  @Test
  public void twoPoliciesOneAppliedOneThrowingIllegalStateExceptionIsNotAppliedAndIsLogged()
      throws PolicyTemplateResolverException {
    IllegalStateException exception =
        new IllegalStateException("More than one directory under ~/runtimePath/.mule/policy-templates/wrongPolicy/META-INF/maven so pom.xml file for artifact in folder ~/runtimePath/.mule/policy-templates/wrongPolicy could not be found");
    when(policyFactory.createFromPolicyDefinition(policyDefinition1)).thenThrow(exception);

    Policy policy = new Policy(mock(PolicyTemplate.class), policyDefinition2, RESOLVED_TEMPLATE);
    when(policyFactory.createFromPolicyDefinition(policyDefinition2))
        .thenReturn(policy);

    PolicySet policySet = new PolicySet(newArrayList(policyDefinition1, policyDefinition2), PLATFORM);
    policiesForApi(API_KEY, policySet);

    List<String> policiesNames = asList(policyDefinition1.getName(), policyDefinition2.getName());

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).isDeploymentFailed(), is(true));
    assertThat(logger.lines(), hasSize(3));
    assertThat(logger.lines().get(0),
               is(new DebugLine("Deploying policies {} from {} to API {}", policiesNames, "Platform", API_KEY)));
    assertThat(logger.lines().get(1),
               is(new ErrorLine("Error deploying policy " + policiesNames.get(0) + " to application " + "test-app", exception)));
    assertThat(logger.lines().get(2), is(new DebugLine("New policy {} detected to apply", policiesNames.get(1))));
    verify(policyDeploymentService).newPolicy(policy);
    verifyNoMoreInteractions(policyDeploymentService);
  }

  private PolicySet policiesWithStatus(DeploymentStatus policy1Status, DeploymentStatus policy2Status) {
    onDeploySetDeploymentStatus(policyDefinition1, policy1, policy1Status);
    onDeploySetDeploymentStatus(policyDefinition2, policy2, policy2Status);

    policySetDeploymentService.addPolicyDeploymentListener(listener);
    return new PolicySet(newArrayList(policyDefinition1, policyDefinition2), PLATFORM);
  }

  private void onDeploySetDeploymentStatus(PolicyDefinition policyDefinition, Policy policy, DeploymentStatus deploymentStatus) {
    doAnswer(invocation -> {
      policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy, deploymentStatus));
      return null;
    }).when(policyDeploymentService).newPolicy(policy);
    doAnswer(invocation -> {
      policyDeploymentTracker.policyRemoved(API_KEY, policyDefinition.getName());
      return null;
    }).when(policyDeploymentService).removePolicy(policy);
    doAnswer(invocation -> {
      policyDeploymentTracker.policyRemoved(API_KEY, policyDefinition.getName());
      policyDeploymentTracker.policyDeployed(API_KEY, new PolicyDeploymentStatus(policy, deploymentStatus));
      return null;
    }).when(policyDeploymentService).updatePolicy(policy);
  }

  private void policiesForApi(ApiKey apiKey, PolicySet policySet) {
    // As now first polling is schedule we must run the retrier.
    policySetDeploymentService.policiesForApi(apiKey, policySet);
    retryTask(0).run();
  }

  private BackoffRunnableRetrier<ApiKey> runnableRetrier(GatewayConfiguration gatewayConfiguration) {
    return new BackoffRunnableRetrier.Builder<ApiKey>("policy-deployment-service", gatewayConfiguration.platformClient()
        .getOutagesStatusCodes(), gatewayConfiguration.platformClient().backoffEnabled())
            .retryUntilNewSchedule()
            .scheduler(backoffSchedulerFactory(), zeroDelayOnScheduling())
            .build();
  }

  private BackoffSchedulerFactory backoffSchedulerFactory() {
    return new FixedExecutorBackoffSchedulerFactory(new ObservableScheduledExecutorService(executorLogger));
  }

  private File definitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/offline-single-api-definition.json").toURI());
  }

  private Runnable retryTask(int index) {
    return executorLogger.scheduledTasks().get(index).runnable();
  }
}
