/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import org.mule.runtime.policy.api.PolicyPointcut;
import org.mule.runtime.policy.api.PolicyPointcutParameters;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Mule's model does not have the concept of API, but Gateway does. In order to implement a pointcut that matches by API, we need
 * an adapter so we can know if an API pointcut matches following Mule's model.
 */
public class ApiPointcutAdapter implements PolicyPointcut {

  private static final Logger LOGGER = LoggerFactory.getLogger(ApiPointcutAdapter.class);

  /**
   * Name of the flow where the API implementation is defined
   */
  private final String flowName;

  /**
   * Name of the policy to which the pointcut belongs
   */
  private final String policyName;

  public ApiPointcutAdapter(String policyName, String flowName) {
    checkNotNull(flowName, "Flow name can not be null");

    this.policyName = policyName;
    this.flowName = flowName;
  }

  @Override
  public boolean matches(PolicyPointcutParameters policyPointcutParameters) {
    checkArgument(!policyPointcutParameters.getComponent().getLocation().getParts().isEmpty(),
                  "Pointcuts parameter must have a component location");

    String flowInExecution =
        policyPointcutParameters.getSourceParameters().orElse(policyPointcutParameters).getComponent().getLocation()
            .getParts().get(0).getPartPath();

    if (!flowName.equals(flowInExecution)) {
      LOGGER.trace("Executing flow {} does not match API's flow {} for policy {}", flowInExecution, flowName, policyName);
      return false;
    }

    LOGGER.trace("Executing flow matches API's flow {} for policy {}", flowName, policyName);
    return true;
  }

}
