/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.lifecyle.healthcheck;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import org.mule.runtime.api.healthcheck.ReadyStatus;

import com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus;

import org.junit.Test;

public class TextAppReadyStatusFactoryTestCase {

  private final ReadyStatusFactory appFactory = new TextReadyStatusFactory();
  private final ReadyStatus readyApi1 = appFactory.readyStatus(1L, GatekeeperStatus.READY);
  private final ReadyStatus readyApi2 = appFactory.readyStatus(2L, GatekeeperStatus.READY);
  private final ReadyStatus untrackedApi1 = appFactory.readyStatus(1L, GatekeeperStatus.UNTRACKED);
  private final ReadyStatus untrackedApi2 = appFactory.readyStatus(2L, GatekeeperStatus.UNTRACKED);

  @Test
  public void oneApiReadyAndOneNot() {
    ReadyStatus readyStatus = appFactory.readyStatus(asList(readyApi1, untrackedApi2));
    assertThat(readyStatus.isReady(), is(false));
    assertThat(readyStatus.statusDescription().isPresent(), is(true));
    assertThat(readyStatus.statusDescription().get(), is(
                                                         "Gatekeeper is blocking the app due to:\n" +
                                                             "API 1: Ready.\n" +
                                                             "API 2: Not Ready. API not found in the API Platform."));
  }

  @Test
  public void orderedStatus() {
    ReadyStatus readyStatus = appFactory.readyStatus(asList(untrackedApi2, readyApi1));
    assertThat(readyStatus.isReady(), is(false));
    assertThat(readyStatus.statusDescription().isPresent(), is(true));
    assertThat(readyStatus.statusDescription().get(), is(
                                                         "Gatekeeper is blocking the app due to:\n" +
                                                             "API 1: Ready.\n" +
                                                             "API 2: Not Ready. API not found in the API Platform."));
  }

  @Test
  public void twoApisReady() {
    ReadyStatus readyStatus = appFactory.readyStatus(asList(readyApi1, readyApi2));
    assertThat(readyStatus.isReady(), is(true));
    assertThat(readyStatus.statusDescription().isPresent(), is(true));
    assertThat(readyStatus.statusDescription().get(), is(
                                                         "API 1: Ready.\n" +
                                                             "API 2: Ready."));
  }

  @Test
  public void twoApisBlocked() {
    ReadyStatus readyStatus = appFactory.readyStatus(asList(untrackedApi1, untrackedApi2));
    assertThat(readyStatus.isReady(), is(false));
    assertThat(readyStatus.statusDescription().isPresent(), is(true));
    assertThat(readyStatus.statusDescription().get(), is(
                                                         "Gatekeeper is blocking the app due to:\n" +
                                                             "API 1: Not Ready. API not found in the API Platform.\n" +
                                                             "API 2: Not Ready. API not found in the API Platform."));
  }

  @Test
  public void singleApiNotReady() {
    ReadyStatus readyStatus = appFactory.readyStatus(singletonList(untrackedApi1));
    assertThat(readyStatus.isReady(), is(false));
    assertThat(readyStatus.statusDescription().isPresent(), is(true));
    assertThat(readyStatus.statusDescription().get(), is("API 1: Not Ready. API not found in the API Platform."));
  }

  @Test
  public void singleApiReady() {
    ReadyStatus readyStatus = appFactory.readyStatus(singletonList(readyApi1));
    assertThat(readyStatus.isReady(), is(true));
    assertThat(readyStatus.statusDescription().isPresent(), is(true));
    assertThat(readyStatus.statusDescription().get(), is("API 1: Ready."));
  }

  @Test
  public void emptyStatusList() {
    ReadyStatus readyStatus = appFactory.readyStatus(emptyList());
    assertThat(readyStatus.isReady(), is(true));
    assertThat(readyStatus.statusDescription().isPresent(), is(true));
    assertThat(readyStatus.statusDescription().get(), is(""));
  }

}
