/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.serialization;

import com.mulesoft.mule.runtime.gw.model.EmptyPolicySpecification;
import com.mulesoft.mule.runtime.gw.model.PolicySpecification;
import com.mulesoft.mule.runtime.gw.model.ValidPolicySpecification;
import org.junit.Test;

import java.io.File;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;

public class PolicySpecificationSerializerTestCase {

  @Test
  public void parseNotExistantYamlFile() {
    File yamlSpec = new File("/xxxxxxx/specs/client-id-enforcement.yaml");
    PolicySpecification policySpecification = new PolicySpecificationBuilder()
        .withFile(yamlSpec)
        .build();

    assertThat(policySpecification, is(instanceOf(EmptyPolicySpecification.class)));
    assertThat(policySpecification.getId(), is(nullValue()));
    assertThat(policySpecification.getName(), is(nullValue()));
    assertThat(policySpecification.getSupportedPoliciesVersions(), is(nullValue()));
    assertThat(policySpecification.getDescription(), is(nullValue()));
    assertThat(policySpecification.getCategory(), is(nullValue()));
    assertThat(policySpecification.getType(), is(nullValue()));
    assertThat(policySpecification.getResourceLevelSupported(), is(nullValue()));
    assertThat(policySpecification.isStandalone(), is(false));
    assertThat(policySpecification.getViolationCategory(), is(nullValue()));
    assertThat(policySpecification.isValid(), is(false));
  }

  @Test
  public void parseValidYaml() {
    File yamlSpec = new File(getClass().getResource("/specs/client-id-enforcement.yaml").getFile());
    PolicySpecification policySpecification = new PolicySpecificationBuilder()
        .withFile(yamlSpec)
        .build();

    assertThat(policySpecification, is(instanceOf(ValidPolicySpecification.class)));
    assertThat(policySpecification.getId(), is("client-id-enforcement"));
    assertThat(policySpecification.getName(), is("Client ID enforcement"));
    assertThat(policySpecification.getSupportedPoliciesVersions(), is(">=v1"));
    assertThat(policySpecification.getDescription(),
               is("All calls to the API must include a client ID and client secret for an application that is registered to use the API.\n\nThis policy will require updates to the RAML definition in order to function. You can obtain the RAML snippet and learn more [here](https://docs.mulesoft.com/anypoint-platform-for-apis/client-id-based-policies).\n"));
    assertThat(policySpecification.getCategory(), is("Compliance"));
    assertThat(policySpecification.getType(), is("system"));
    assertThat(policySpecification.getResourceLevelSupported(), is("true"));
    assertThat(policySpecification.isStandalone(), is(true));
    assertThat(policySpecification.getViolationCategory(), is("authentication"));
    assertThat(policySpecification.isValid(), is(true));
    assertThat(policySpecification.getConfiguration().size(), is(3));
  }

  @Test
  public void parseValidYamlWithIdentityManagement() {
    File yamlSpec = new File(getClass().getResource("/specs/openidconnect-access-token-enforcement.yaml").getFile());
    PolicySpecification policySpecification = new PolicySpecificationBuilder()
        .withFile(yamlSpec)
        .build();

    assertThat(policySpecification, is(instanceOf(ValidPolicySpecification.class)));
    assertThat(policySpecification.getId(), is("openidconnect-access-token-enforcement"));
    assertThat(policySpecification.getName(), is("OpenId Connect access token enforcement"));
    assertThat(policySpecification.getSupportedPoliciesVersions(), is(">=v4"));
    assertThat(policySpecification.getDescription(),
               is("Enforces access tokens by OpenId Connect.\n\nThis policy will require updates to the RAML/OAS definition in order to function.\nYou can obtain the RAML snippet and learn more [here](https://github.com/raml-org/raml-spec/blob/master/versions/raml-08/raml-08.md#security).\nYou can obtain OAS snippet and learn more [here](https://swagger.io/specification/v2/#securityRequirementObject).\n"));
    assertThat(policySpecification.getCategory(), is("Security"));
    assertThat(policySpecification.getType(), is("system"));
    assertThat(policySpecification.getResourceLevelSupported(), is("true"));
    assertThat(policySpecification.isStandalone(), is(true));
    assertThat(policySpecification.getViolationCategory(), is("authentication"));
    assertThat(policySpecification.isValid(), is(true));
    assertThat(policySpecification.getIdentityManagement().get().getType(), is("OpenIDConnect"));
    assertThat(policySpecification.getConfiguration().size(), is(3));
  }

  @Test
  public void parseInvalidYaml() {
    File yamlSpec = new File(getClass().getResource("/specs/invalid.yaml").getFile());

    PolicySpecification policySpecification = new PolicySpecificationBuilder()
        .withFile(yamlSpec)
        .build();

    assertThat(policySpecification, is(instanceOf(EmptyPolicySpecification.class)));
    assertThat(policySpecification.getId(), is(nullValue()));
    assertThat(policySpecification.getName(), is(nullValue()));
    assertThat(policySpecification.getSupportedPoliciesVersions(), is(nullValue()));
    assertThat(policySpecification.getDescription(), is(nullValue()));
    assertThat(policySpecification.getCategory(), is(nullValue()));
    assertThat(policySpecification.getType(), is(nullValue()));
    assertThat(policySpecification.getResourceLevelSupported(), is(nullValue()));
    assertThat(policySpecification.isStandalone(), is(false));
    assertThat(policySpecification.getViolationCategory(), is(nullValue()));
    assertThat(policySpecification.isValid(), is(false));
  }

  @Test
  public void parsePartiallyValidYaml() {
    File yamlSpec = new File(getClass().getResource("/specs/partial.yaml").getFile());
    PolicySpecification validPolicySpecification = new PolicySpecificationBuilder()
        .withFile(yamlSpec)
        .build();

    assertThat(validPolicySpecification.getId(), is("client-id-enforcement"));
    assertThat(validPolicySpecification.getName(), is("Client ID enforcement"));
    assertThat(validPolicySpecification.getSupportedPoliciesVersions(), is(nullValue()));
    assertThat(validPolicySpecification.getDescription(), is(nullValue()));
    assertThat(validPolicySpecification.getCategory(), is("Compliance"));
    assertThat(validPolicySpecification.getType(), is("system"));
    assertThat(validPolicySpecification.getResourceLevelSupported(), is("true"));
    assertThat(validPolicySpecification.isStandalone(), is(true));
    assertThat(validPolicySpecification.getViolationCategory(), is(nullValue()));
    assertThat(validPolicySpecification.isValid(), is(true));
    assertThat(validPolicySpecification.getConfiguration(), is(notNullValue()));
  }
}
