/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.deployment.TransactionalPolicyDeployer;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore;
import com.mulesoft.mule.runtime.gw.policies.store.EncryptedPropertiesSerializer;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Answers;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class PolicyDeploymentServiceTestCase extends AbstractMuleTestCase {

  private static final String RESOLVED_TEMPLATE = "resolvedTemplate";

  @Rule
  public SystemPropertyTemporaryFolder temporaryFolder = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  @Mock
  private TransactionalPolicyDeployer policyDeployer;

  @Mock
  private ApiService apiService;

  @Mock(answer = Answers.RETURNS_DEEP_STUBS)
  private Api api;

  private Policy policy;
  private PolicyDefinition policyDefinition;
  private PolicyDeploymentService policyDeploymentService;
  private PolicyDeploymentTracker policyDeploymentTracker;

  @Mock
  private PolicyFactory policyFactory;

  @Mock
  private PolicyTemplate policyTemplate;

  @Before
  public void setUp() {
    when(apiService.get(API_KEY)).thenReturn(of(api));

    PolicyStore policyStore = new DefaultPolicyStore(new EncryptedPropertiesSerializer());

    policyDeploymentTracker = new DefaultPolicyDeploymentTracker();
    policyDeploymentService = new DefaultPolicyDeploymentService(apiService,
                                                                 policyDeploymentTracker,
                                                                 policyDeployer);

    overrideVariable("policyDeployer").in(policyDeploymentService).with(policyDeployer);

    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(emptyMap()));
    policy = new Policy(policyTemplate, policyDefinition, RESOLVED_TEMPLATE);
    when(policyFactory.createFromPolicyDefinition(policyDefinition)).thenReturn(policy);
  }

  @Test
  public void newPolicySingleApi() {
    policyDeploymentService.newPolicy(policyDefinition);

    verify(policyDeployer).deploy(policyDefinition, api);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void newPolicyMultipleApis() {
    ApiImplementation implementation2 = mock(ApiImplementation.class);
    Api api2 = mock(Api.class);
    when(api2.getImplementation()).thenReturn(implementation2);
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, newArrayList(API_KEY, API_KEY_2), null, 1,
                             new PolicyConfiguration(emptyMap()));
    when(apiService.get(API_KEY_2)).thenReturn(of(api2));
    policy = new Policy(policyTemplate, policyDefinition, RESOLVED_TEMPLATE);

    policyDeploymentService.newPolicy(policyDefinition);

    verify(policyDeployer).deploy(policyDefinition, api);
    verify(policyDeployer).deploy(policyDefinition, api2);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void newPolicyForSpecificNotExistingApi() {
    when(apiService.get(API_KEY_2)).thenReturn(empty());
    policyDefinition = new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, newArrayList(API_KEY_2), null, 1,
                                            new PolicyConfiguration(emptyMap()));
    policyDeploymentService.newPolicy(policyDefinition);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void policyRemoved() {

    policyDeploymentService.removePolicy(policyDefinition);

    verify(policyDeployer).undeploy(policyDefinition.getName(), api);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void multipleApisPolicyRemoved() {
    ApiImplementation implementation2 = mock(ApiImplementation.class);
    Api api2 = mock(Api.class);
    when(api2.getImplementation()).thenReturn(implementation2);
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, newArrayList(API_KEY, API_KEY_2), null, 1,
                             new PolicyConfiguration(emptyMap()));
    policy = new Policy(policyTemplate, policyDefinition, RESOLVED_TEMPLATE);
    when(apiService.get(API_KEY_2)).thenReturn(of(api2));

    policyDeploymentService.removePolicy(policyDefinition);

    verify(policyDeployer).undeploy(policyDefinition.getName(), api);
    verify(policyDeployer).undeploy(policyDefinition.getName(), api2);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void policyRemovedByName() {
    policyDeploymentTracker.policyDeployed(API_KEY,
                                           new PolicyDeploymentStatus(new PolicyDefinitionDeploymentStatus(policyDefinition)));
    policyDeploymentService.removePolicy(policyDefinition.getName());

    verify(policyDeployer).undeploy(policyDefinition.getName(), api);
    verifyNoMoreInteractions(policyDeployer);
  }

  @Test
  public void policyUpdated() {
    PolicyDefinition updatedDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 2, new PolicyConfiguration(emptyMap()));

    policyDeploymentService.updatePolicy(policyDefinition, updatedDefinition);
    verify(policyDeployer).update(policyDefinition, updatedDefinition, api);
    verifyNoMoreInteractions(policyDeployer);
  }
}
