/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies;

import static java.util.Collections.singletonList;

import com.mulesoft.mule.runtime.gw.api.construction.Builder;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class PolicyDefinitionBuilder implements Builder<PolicyDefinition> {

  private final PolicyDefinition baseDefinition;
  private int order;
  private List<ApiKey> apiKey;
  private PolicyConfiguration configuration;

  private static final String IDENTITY_MANAGEMENT_TOKEN_URL = "identityManagementTokenUrl";
  private static final String IDENTITY_MANAGEMENT_CLIENT_ID = "identityManagementClientId";
  private static final String IDENTITY_MANAGEMENT_CLIENT_SECRET = "identityManagementClientSecret";

  public static boolean hasIdentityManagementProperties(PolicyDefinition policyDefinition) {
    Map<String, Object> configuration = policyDefinition.getConfigurationData().getConfiguration();
    return configuration.get(IDENTITY_MANAGEMENT_TOKEN_URL) != null
        || configuration.get(IDENTITY_MANAGEMENT_CLIENT_ID) != null
        || configuration.get(IDENTITY_MANAGEMENT_CLIENT_SECRET) != null;
  }

  public PolicyDefinitionBuilder(PolicyDefinition baseDefinition) {
    this.baseDefinition = baseDefinition;
    this.order = baseDefinition.getOrder();
    this.apiKey = baseDefinition.getApiKeys();
    configuration = baseDefinition.getConfigurationData();
  }

  public PolicyDefinitionBuilder order(int order) {
    this.order = order;
    return this;
  }

  public PolicyDefinitionBuilder apiKey(ApiKey apiKey) {
    this.apiKey = singletonList(apiKey);
    return this;
  }

  public PolicyDefinitionBuilder noIdentityManagement() {
    configuration = new PolicyConfiguration(configuration.getConfiguration()
        .entrySet()
        .stream()
        .filter(entry -> !isIdentityProperty(entry.getKey()))
        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    return this;
  }

  private boolean isIdentityProperty(String key) {
    return key.equals(IDENTITY_MANAGEMENT_TOKEN_URL) || key.equals(IDENTITY_MANAGEMENT_CLIENT_ID)
        || key.equals(IDENTITY_MANAGEMENT_CLIENT_SECRET);
  }

  @Override
  public PolicyDefinition build() {
    if (baseDefinition instanceof OfflinePolicyDefinition) {
      return new OfflinePolicyDefinition(baseDefinition.getId(), baseDefinition.getTemplateKey(),
                                         apiKey, baseDefinition.getHttpResourcePointcuts(), order,
                                         configuration);
    } else {
      return new PolicyDefinition(baseDefinition.getId(), baseDefinition.getTemplateKey(),
                                  apiKey, baseDefinition.getHttpResourcePointcuts(), order,
                                  configuration);
    }
  }
}
