/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.deployment;

import static java.util.Collections.emptyList;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;

import org.mule.runtime.core.api.policy.PolicyParametrization;

import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyParametrizationFactory;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Deploys policies that are used for internal purposes. Deployment and undeployment of the policies are executed with the API's
 * implementation classloader
 */
public class InternalPolicyDeployer implements PolicyDeployer {

  private static final Logger LOGGER = LoggerFactory.getLogger(InternalPolicyDeployer.class);

  private final PolicyFactory policyFactory;
  private final PolicyParametrizationFactory parametrizationFactory;

  public InternalPolicyDeployer(PolicyFactory policyFactory) {
    this.policyFactory = policyFactory;
    this.parametrizationFactory = new PolicyParametrizationFactory();
  }

  @Override
  public PolicyDefinitionDeploymentStatus deploy(PolicyDefinition policyDefinition, Api api) {
    ApiImplementation apiImplementation = api.getImplementation();
    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);
    withContextClassLoader(
                           apiImplementation.getArtifactClassloader(),
                           () -> {
                             LOGGER.debug("Deploying internal policy {} for {}", policyDefinition.getName(), api);

                             try {
                               PolicyTemplate policyTemplate = policy.getPolicyTemplate();

                               PolicyParametrization policyParametrization =
                                   parametrizationFactory.create(policyDefinition, apiImplementation,
                                                                 policyTemplate.getTemplateFile(), null, false, emptyList());

                               apiImplementation.addPolicy(policyTemplate.getTemplateDescriptor(), policyParametrization);
                             } catch (Exception e) {
                               LOGGER.error("Error deploying internal policy {} to API {}: {}", policyDefinition.getName(),
                                            apiImplementation.getApiKey(),
                                            e.getMessage());
                               throw new InternalPolicyDeploymentException("Unexpected error deploying internal policy", e);
                             }
                           });
    return new PolicyDefinitionDeploymentStatus(policyDefinition);
  }

  @Override
  public PolicyDefinitionDeploymentStatus updateOrder(PolicyDefinition policy, Api api) {
    throw new InternalPolicyDeploymentException("Internal policies should not be reordered");
  }

  @Override
  public boolean undeploy(String policyName, Api api) {
    ApiImplementation apiImplementation = api.getImplementation();
    return withContextClassLoader(
                                  apiImplementation.getArtifactClassloader(),
                                  () -> {
                                    LOGGER.debug("Removing internal policy {} from {}", policyName, api);

                                    return apiImplementation
                                        .removePolicy(parametrizationFactory.buildParametrizationId(policyName,
                                                                                                    apiImplementation));
                                  });
  }
}
