/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.encryption.encrypter;

import static java.lang.String.join;
import static java.util.stream.Collectors.toMap;

import com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypter;
import com.mulesoft.mule.runtime.gw.model.PolicyProperty;
import com.mulesoft.mule.runtime.gw.policies.encryption.EncryptedValueResult;
import com.mulesoft.mule.runtime.gw.policies.encryption.filter.ListOfKeyvaluesTypeFilter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Knows how to encrypt and decrypt multiple config values with type keyvalue
 */
public class ListOfKeyvaluesEncrypter extends PolicyTypeEncrypter<List<Map<String, Object>>> {

  public ListOfKeyvaluesEncrypter(RuntimeEncrypter runtimeEncrypter, boolean isSensitiveOnlyEnabled,
                                  List<PolicyProperty> policyProperties) {
    super(runtimeEncrypter, isSensitiveOnlyEnabled, policyProperties);
  }

  @Override
  protected EncryptedValueResult encryptSingle(String rootKey, List<Map<String, Object>> keyvalues) {
    List<Map<String, Object>> templateList = new ArrayList<>();
    List<Map<String, Object>> configurationList = new ArrayList<>();

    flatten(keyvalues).forEach((key, value) -> {
      String templateValue = join("", TEMPLATE_PREFIX, rootKey, ".", key, TEMPLATE_SUFFIX);
      String configurationValue = encrypt((String) value);

      Map<String, Object> templateMap = new LinkedHashMap<>();
      templateMap.put("key", key);
      templateMap.put("value", templateValue);

      Map<String, Object> configurationMap = new LinkedHashMap<>();
      configurationMap.put("key", key);
      configurationMap.put("value", configurationValue);

      templateList.add(templateMap);
      configurationList.add(configurationMap);
    });

    Map<String, Object> configFile = flatten(configurationList);

    return new EncryptedValueResult<>(rootKey, templateList, configurationList, configFile);
  }

  @Override
  protected EncryptedValueResult decryptSingle(String key, List<Map<String, Object>> keyvalues) {
    List<Map<Object, Object>> updated = keyvalues.stream().map(keyvalue -> {
      Map<Object, Object> decrypted = new HashMap<>();
      decrypted.put("key", keyvalue.get("key"));
      decrypted.put("value", decrypt((String) keyvalue.get("value")));
      return decrypted;
    }).collect(Collectors.toList());

    return new EncryptedValueResult<>(key, updated, updated);
  }

  @Override
  protected boolean supports(String key, Optional<PolicyProperty> property) {
    return property.isPresent() && new ListOfKeyvaluesTypeFilter().test(property.get());
  }

  private Map<String, Object> flatten(List<Map<String, Object>> keyvalues) {
    return keyvalues.stream()
        .collect(toMap(stringObjectEntry1 -> (String) stringObjectEntry1.get("key"),
                       stringObjectEntry12 -> stringObjectEntry12.get("value")));
  }
}
