/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.factory;

import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.encryption.PolicyConfigurationEncrypter;
import com.mulesoft.mule.runtime.gw.policies.encryption.PolicyConfigurationEncryptionResult;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;
import com.mulesoft.mule.runtime.gw.policies.template.provider.PolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.resolver.PolicyTemplateResolver;

import java.util.Map;

/**
 * Factory for {@link Policy}
 */
public class EncryptedPolicyFactory extends PolicyFactoryTemplate {

  private PolicyFactory defaultPolicyFactory;
  private PolicyConfigurationEncrypter policyConfigurationEncrypter;

  public EncryptedPolicyFactory(PolicyTemplateResolver templateResolver,
                                PolicyTemplateProvider templateProvider,
                                PolicyConfigurationEncrypter policyConfigurationEncrypter) {
    super(templateResolver, templateProvider);
    this.defaultPolicyFactory = new DefaultPolicyFactory(templateResolver, templateProvider);
    this.policyConfigurationEncrypter = policyConfigurationEncrypter;
  }

  @Override
  public Policy createFromPolicyDefinition(PolicyDefinition policyDefinition) {

    PolicyTemplate policyTemplate = getPolicyTemplate(policyDefinition);

    if (policyTemplate.getPolicySpecification().isEncryptionSupported()) {
      return createFromPolicyDefinitionEncrypted(policyTemplate, policyDefinition);
    }

    return defaultPolicyFactory.createFromPolicyDefinition(policyDefinition);
  }

  private Policy createFromPolicyDefinitionEncrypted(PolicyTemplate policyTemplate, PolicyDefinition policyDefinition) {
    policyDefinition = evaluatePolicyIdentityManagement(policyDefinition, policyTemplate);

    PolicyConfigurationEncryptionResult policyConfigurationEncryptionResult =
        policyConfigurationEncrypter.encryptConfiguration(policyDefinition,
                                                          policyTemplate.getPolicySpecification());

    Map<String, Object> policyDefinitionConfiguration = policyConfigurationEncryptionResult.getTemplatePlaceholders();


    String resolvedTemplate =
        resolveTemplate(policyTemplate, policyDefinitionConfiguration);

    PolicyDefinition encryptedPolicyDefinition;
    if (policyDefinition.isOnline()) {
      encryptedPolicyDefinition = new PolicyDefinition(policyDefinition,
                                                       resolvePolicyConfiguration(policyDefinition,
                                                                                  policyConfigurationEncryptionResult));
    } else {
      encryptedPolicyDefinition = new OfflinePolicyDefinition(policyDefinition,
                                                              resolvePolicyConfiguration(policyDefinition,
                                                                                         policyConfigurationEncryptionResult));
    }

    return new Policy(policyTemplate, encryptedPolicyDefinition, resolvedTemplate,
                      policyConfigurationEncryptionResult.getConfigFileProperties());
  }

  private PolicyConfiguration resolvePolicyConfiguration(PolicyDefinition originalPolicyDefinition,
                                                         PolicyConfigurationEncryptionResult policyConfigurationEncryptionResult) {
    return new PolicyConfiguration(policyConfigurationEncryptionResult.getConfigurationProperties().getConfiguration(),
                                   originalPolicyDefinition.getConfigurationData().getConfigurationVersion());
  }

}
