/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.lifecyle;

import static java.util.Collections.emptyMap;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;

import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.context.notification.MuleContextNotification;
import org.mule.runtime.core.api.context.notification.MuleContextNotificationListener;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatusTracker;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.deployment.InternalPolicyDeployer;
import com.mulesoft.mule.runtime.gw.policies.deployment.PolicyDeployer;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;

import java.util.ArrayList;

import org.slf4j.Logger;

/**
 * GateKeeper can operate in three different modes:
 *
 * - STRICT (default): blocks the API till the first policy reconciliation cycle. If the request succeeds and all te policies are
 * applied OK, then the API is unblocked. Otherwise the API remains blocked till a successful reconciliation cycle.
 *
 * - FLEXIBLE: blocks the API till the first policy reconciliation cycle. If the request succeeds and all the policies are applied
 * OK, then the API is unblocked. If the request fails with one of the outages status codes and a cached version exists, then the
 * policies from the cached request are applied and if successful the API is unblocked. Otherwise the API remains blocked till a
 * successful reconciliation cycle.
 *
 * - DISABLED: the API is never blocked, so it can be accessed even before any policy is applied.
 *
 * <p>
 * Once an app tracked flows are enabled they remain so even if a future policy application fails.
 * <p>
 * GateKeeper logic:
 * <p>
 * at gateway startup - remove local online policies from filesystem
 * <p>
 * at app deploy - stop app tracked flows on deploy
 * <p>
 * at API policies request time - cache response if successful
 * <p>
 * at policy store update (from polling or HC notification) - update required policies for particular API - check if any blocked
 * API can be enabled
 * <p>
 * at policy application/removal - if successfully applied or removed update applied policy list - assert blocked APIs required
 * policies and start those with nothing pending
 */
public abstract class BlockingGateKeeper extends GateKeeper {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(BlockingGateKeeper.class);

  private static final PolicyTemplateKey GATEKEEPER_POLICY_TEMPLATE_KEY =
      new PolicyTemplateKey("com.mulesoft.anypoint", "mule-gatekeeper-policy-template", getProductVersion());
  private static final String GATEKEEPER_POLICY_ID = "gatekeeper";

  private final PolicyDeployer policyDeployer;

  public BlockingGateKeeper(ApiService apiService,
                            PolicyFactory policyFactory) {
    super(apiService);
    this.policyDeployer = new InternalPolicyDeployer(policyFactory);
  }

  /**
   * Block API because it is being undeployed or because it is now untracked
   *
   * @param apiKey key of the API
   */
  @Override
  public void onPoliciesRemoved(ApiKey apiKey) {
    getMonitoredApi(apiKey).ifPresent(this::doBlockApi);
  }

  @Override
  protected boolean isMonitored(Api api) {
    return !api.getImplementation().isHdp();
  }

  @Override
  protected void blockApi(Api api) {
    MuleContext context = api.getImplementation().getFlow().getMuleContext();
    context.getNotificationManager().addListener(new BlockingGateKeeper.GateKeeperMuleContextListener(api, this));
  }

  protected void doBlockApi(Api api) {
    ApiImplementation implementation = api.getImplementation();
    GatekeeperStatusTracker gatekeeperStatus = implementation.gatekeeperStatus();
    if (!gatekeeperStatus.isBlocked()) {
      PolicyDefinition policyDefinition =
          new OfflinePolicyDefinition(getPolicyId(api), GATEKEEPER_POLICY_TEMPLATE_KEY, implementation.getApiKey(),
                                      new ArrayList<>(), 1, new PolicyConfiguration(emptyMap()));

      policyDeployer.deploy(policyDefinition, api);
      gatekeeperStatus.blocked();

      LOGGER.info("API {} is blocked (unavailable).", implementation.getApiKey());
    }
  }

  @Override
  protected void unblockApi(Api api) {
    ApiImplementation implementation = api.getImplementation();
    policyDeployer.undeploy(getPolicyId(api), api);

    implementation.gatekeeperStatus().unblocked();

    LOGGER.info("API {} is now unblocked (available).", implementation.getApiKey());
  }


  private String getPolicyId(Api api) {
    return GATEKEEPER_POLICY_ID + "-" + api.getKey().id();
  }

  /**
   * MuleContext listener used to block an API implementation before it starts its listener. It does that by attaching to the
   * CONTEXT_STARTING event of MuleContext notification that occurs just before context start is fired
   */
  public static class GateKeeperMuleContextListener implements MuleContextNotificationListener<MuleContextNotification> {

    private final Api api;
    private final BlockingGateKeeper gateKeeper;

    public GateKeeperMuleContextListener(Api api,
                                         BlockingGateKeeper gateKeeper) {
      this.api = api;
      this.gateKeeper = gateKeeper;
    }

    @Override
    public void onNotification(MuleContextNotification notification) {
      if (isMuleContextStarting(notification)) {
        gateKeeper.doBlockApi(api);
      } else if (isMuleContextStarted(notification)) {
        api.getImplementation().getFlow().getMuleContext().getNotificationManager().removeListener(this);
      }
    }

    private boolean isMuleContextStarting(MuleContextNotification notification) {
      return MuleContextNotification.CONTEXT_STARTING == notification.getAction().getActionId();
    }

    private boolean isMuleContextStarted(MuleContextNotification notification) {
      return MuleContextNotification.CONTEXT_STARTED == notification.getAction().getActionId();
    }
  }

}
