/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.lifecyle;

import org.mule.runtime.api.healthcheck.HealthCheckValidator;
import org.mule.runtime.api.healthcheck.ReadyStatus;

import com.mulesoft.mule.runtime.gw.api.agent.HealthCheck;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.healthcheck.ReadyStatusFactory;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.healthcheck.TextReadyStatusFactory;

import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultHealthCheck implements HealthCheck {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultHealthCheck.class);
  private final ReadyStatusFactory readyStatusFactory;

  private ApiService apiService;

  public DefaultHealthCheck(ApiService apiService) {
    this.apiService = apiService;
    this.readyStatusFactory = new TextReadyStatusFactory();
  }

  public boolean isInstanceTrafficAllowed() {
    boolean trafficAllowed = apiService.getApis().stream()
        .noneMatch(api -> api.getImplementation().gatekeeperStatus().isBlocked());

    if (LOGGER.isDebugEnabled()) {
      apiService.getApis().stream().map(api -> api.getImplementation().getArtifactName())
          .collect(Collectors.toSet())
          .forEach(this::applicationTrafficAllowed);
    }
    return trafficAllowed;
  }

  protected ReadyStatus isApiTrafficAllowed(Long apiId) {
    return apiService
        .getApis()
        .stream()
        .filter(api -> api.getKey().id().equals(apiId))
        .findFirst()
        .map(api -> readyStatusFactory.readyStatus(apiId, api.getImplementation().gatekeeperStatus().status()))
        .orElseGet(() -> readyStatusFactory.missingApi(apiId));
  }

  protected ReadyStatus applicationTrafficAllowed(String applicationName) {
    List<ReadyStatus> statusList = apiService.getApis().stream()
        .filter(api -> api.getImplementation().getArtifactName().equals(applicationName))
        .map(api -> readyStatusFactory.readyStatus(api.getKey().id(), api.getImplementation().gatekeeperStatus().status()))
        .collect(Collectors.toList());

    ReadyStatus status = readyStatusFactory.readyStatus(statusList);
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("App {}: {}.", applicationName, status.isReady() ? "Ready" : "Blocked");
      status.statusDescription().ifPresent(LOGGER::debug);
    }
    return status;
  }

  @Override
  public boolean isApplicationTrafficAllowed(String applicationName) {
    return applicationTrafficAllowed(applicationName).isReady();
  }

  public HealthCheckValidator getValidator(String applicationName) {
    return () -> applicationTrafficAllowed(applicationName);
  }

  public HealthCheckValidator getApiValidator(Long apiId) {
    return () -> isApiTrafficAllowed(apiId);
  }
}
